      SUBROUTINE GETSTA_PTS ( M_PTS, L_PTS, STAT_NAME, I_TYP, LBUF, BUF, &
     &                        TIME_ARR, VAL_ARR, SIG_ARR, MODU_ARR, MODC_ARR, &
     &                        IUER )
! ************************************************************************
! *                                                                      *
! *   Routine  GETSTA_PTS  parses the text buffer with station-dependent *
! *   plot information about clock and atmosphere or EOP. It is assumed  *
! *   that the file with plot information generated by ADJST has been    *
! *   read and its content is in the buffer BUF.                         *
! *                                                                      *
! * _________________________ Input parameters: ________________________ *
! *                                                                      *
! *      M_PTS ( INTEGER*4 ) -- Maximal number of values to be extracted *
! *  STAT_NAME ( CHARACTER ) -- Array of stations names.                 *
! *                             Dimension: L_STA                         *
! *      I_TYP ( INTEGER*4 ) -- Type of plotting values. The following   *
! *                             types are supported (defined in mdlcm.i) *
! *                          ICLP_TYP -- "Piese-wise clock function"     *
! *                          ICLT_TYP -- "Total clock function"          *
! *                          ICLB_TYP -- "clock function with breaks"    *
! *                          IATP_TYP -- "Piese-wise atmosphere path     *
! *                                       delay"                         *
! *                          IXPL_TYP -- "Piese-wise X pole coordinate"  *
! *                          IYPL_TYP -- "Piese-wise Y pole coordinate"  *
! *                          IUT1_TYP -- "Piese-wise UT1 arguments"      *
! *       LBUF ( INTEGER*4 ) -- Number of lines in the buffer            *
! *        BUF ( CHARACTER ) -- Buffer with content of plot file         *
! *                             generated by ADJST.                      *
! *                                                                      *
! * ________________________ Output parameters: ________________________ *
! *                                                                      *
! *      L_PTS ( INTEGER*4 ) -- Number of extracted values (number of    *
! *                             time epochs).                            *
! *   TIME_ARR ( REAL*8    ) -- Array of arguments: time epochs in hours.*
! *                             Dimension: L_PTS.                        *
! *    VAL_ARR ( REAL*8    ) -- Array of values. Units: psec for delay,  *
! *                             microseconds of time for UT1,            *
! *                             microarcseconds for pole coordinates.    *
! *                             Dimension: L_PTS.                        *
! *    SIG_ARR ( REAL*8    ) -- Array of formal uncertainties. Units are *
! *                             the same as for VAL_ARR.                 *
! *                             Dimension: L_PTS.                        *
! *   MODU_ARR ( REAL*8    ) -- Array of a priori values of EOP (zero    *
! *                             for other types). Value of a priori EOP  *
! *                             applyed before estimation plus           *
! *                             a constant shift found to make           *
! *                             the difference between estimates and     *
! *                             a priori value at the moment of the      *
! *                             first epoch to be zero.                  *
! *   MODC_ARR ( REAL*8    ) -- Array of model values of EOP (zero for   *
! *                             other types). It is ued for comparison   *
! *                             purpose only. A constant shift is        *
! *                             applied to all mopdel values in order to *
! *                             make the difference between estimates    *
! *                             and a priori value at the moment of the  *
! *                             first epoch to be zero.                  *
! *                                                                      *
! * ________________________ Modified parameters: ______________________ *
! *                                                                      *
! *         IUER ( INTEGER*4, OPT ) -- Universal error handler.          *
! *                           Input: switch IUER=0 -- no error messages  *
! *                                  will be generated even in the case  *
! *                                  of error. IUER=-1 -- in the case of *
! *                                  error the message will be put on    *
! *                                  stdout.                             *
! *                           Output: 0 in the case of successful        *
! *                                   completion and non-zero in the     *
! *                                   case of error.                     *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ###  31-OCT-97   GETSTA_PTS   v1.4  (d)  L. Petrov 17-JUL-2000 ###  *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE ! Updated by Jim Ryan for I*4 compliance, Sept 2002
      INCLUDE    'solve.i'
      INCLUDE    'mdlcm.i'
      INTEGER*4  M_PTS, L_PTS, I_TYP, LBUF, IUER
      CHARACTER  STAT_NAME*(*), BUF(LBUF)*(*)
      REAL*8     TIME_ARR(M_PTS), VAL_ARR(M_PTS), SIG_ARR(M_PTS), &
     &           MODU_ARR(M_PTS), MODC_ARR(M_PTS), JD, JD_FIRST, DUT1, DX, DY, &
     &           MODC_SHIFT, MODU_SHIFT
      CHARACTER  REG*3, WORD*128
      INTEGER*4  MIND, LIND
      PARAMETER  ( MIND = 1024 )
      INTEGER*4  IND(2,MIND), J1, J2, IC, ICOL, IBG, IEN, IER, ICT
      INTEGER*4  I_LEN
!
! --- Parse the first line
!
      REG = CHAR(0)//CHAR(32)//CHAR(9)
      CALL EXWORD ( BUF(1), MIND, LIND, IND, REG, -3 )
      IF ( LIND .LT. 8 ) THEN
           CALL ERR_LOG ( 9231, IUER, 'GETSTA_PTS', 'Too few words at the '// &
     &                   'first line ' )
           RETURN
      END IF
!
      IC    = 0
      ICOL  = 0
      L_PTS = 0
!
! --- Extract the station name. Learn the number of columnns ( ICOL ) and
! --- the column index with time tag ( ICT )
!
      IF ( I_TYP .EQ. ICLP_TYP  .OR.  I_TYP .EQ. ICLT_TYP  .OR. &
     &     I_TYP .EQ. ICLB_TYP  .OR.  I_TYP .EQ. IATP_TYP        ) THEN
           DO 410 J1=8,LIND
              IC = IC + 1
!
! ----------- Special trick for names with blanks
!
              IF ( ( IND(1,J1) - IND(1,J1-1) ) .LT. 8 ) IC = IC -1
              CALL CLRCH ( WORD )
              WORD = BUF(1)(IND(1,J1):IND(1,J1)+7)
              IF ( WORD .EQ. STAT_NAME ) THEN
                   ICOL = IC
              END IF
 410       CONTINUE
!
           IF ( ICOL .EQ. 0 ) THEN
                CALL ERR_LOG ( 9232, IUER, 'GETSTA_PTS', 'Station '// &
     &               STAT_NAME(1:I_LEN(STAT_NAME))//' has not been '// &
     &              'found at the first line' )
                RETURN
           END IF
           ICT  = 6 ! Column number with time tag
         ELSE IF ( I_TYP .EQ. IXPL_TYP ) THEN
           ICT  = 1
           ICOL = 1
         ELSE IF ( I_TYP .EQ. IYPL_TYP ) THEN
           ICT  = 1
           ICOL = 2
         ELSE IF ( I_TYP .EQ. IUT1_TYP ) THEN
           ICT  = 1
           ICOL = 3
      END IF
!
! --- Cycle over rows of the buffer
!
      L_PTS = 0
      DO 420 J2=4,LBUF
!
! ------ Parse the line onto words
!
         CALL EXWORD   ( BUF(J2), MIND, LIND, IND, REG, -3 )
         IF ( LIND .LT. 6 + 2*ICOL ) GOTO 420 ! too few words
!
         L_PTS = L_PTS + 1
!
! ------ Reading time tag
!
         CALL DFOR_MEN ( BUF(J2)(IND(1,ICT):IND(2,ICT)), JD, IER )
         IF ( IER .NE. 0 ) THEN
              CALL ERR_LOG ( 9233, IUER, 'GETSTA_PTS', 'Error reading '// &
     &            'time tag at the -- '//BUF(J2)(IND(1,ICT):IND(2,ICT)) )
              RETURN
         END IF
         IF ( L_PTS .EQ. 1 ) JD_FIRST = JD
         TIME_ARR(L_PTS) = ( JD - JD_FIRST )*24.0
!
! ------ Reading the value of the parameter
!
         IF ( I_TYP .EQ. ICLP_TYP  .OR.  I_TYP .EQ. ICLT_TYP  .OR. &
     &        I_TYP .EQ. ICLB_TYP  .OR.  I_TYP .EQ. IATP_TYP       ) THEN
!
              IBG = 31  + 21*(ICOL-1)
              IEN = IBG + 14
           ELSE IF ( I_TYP .EQ. IXPL_TYP   .OR.   I_TYP .EQ. IYPL_TYP  .OR. &
     &               I_TYP .EQ. IUT1_TYP  ) THEN
              IBG = 41  + 39*(ICOL-1)
              IEN = IBG + 12
         END IF
         IF ( BUF(J2)(IBG+1:IBG+1) .EQ. '$' ) THEN
!
! ----------- Dollars means "no information for this station". Skip this line
!
              L_PTS = L_PTS - 1
              GOTO 420
         END IF
         CALL DFOR_MEN ( BUF(J2)(IBG:IEN), VAL_ARR(L_PTS), IER )
         IF ( IER .NE. 0 ) THEN
              CALL ERR_LOG ( 9234, IUER, 'GETSTA_PTS', 'Error reading '// &
     &            'value of the ploted parameter: '//BUF(J2)(IBG:IEN) )
              RETURN
         END IF
!
! ------ Reading the uncertainty of the parameter
!
         IF ( I_TYP .EQ. ICLP_TYP  .OR.  I_TYP .EQ. ICLT_TYP  .OR. &
     &        I_TYP .EQ. ICLB_TYP  .OR.  I_TYP .EQ. IATP_TYP       ) THEN
!
              IBG = 46  + 21*(ICOL-1)
              IEN = IBG + 5
           ELSE IF ( I_TYP .EQ. IXPL_TYP   .OR.   I_TYP .EQ. IYPL_TYP  .OR. &
     &               I_TYP .EQ. IUT1_TYP  ) THEN
              IBG = 54  + 39*(ICOL-1)
              IEN = IBG + 12
         END IF
         CALL DFOR_MEN ( BUF(J2)(IBG:IEN), SIG_ARR(L_PTS), IER )
         IF ( IER .NE. 0 ) THEN
              CALL ERR_LOG ( 9235, IUER, 'GETSTA_PTS', 'Error reading '// &
     &            'uncertainty of the ploted parameter: '//BUF(J2)(IBG:IEN) )
              RETURN
         END IF
         IF ( I_TYP .EQ. ICLT_TYP ) THEN
              VAL_ARR(L_PTS) = VAL_ARR(L_PTS)*1.D-3
              SIG_ARR(L_PTS) = SIG_ARR(L_PTS)*1.D-3
         END IF
!
! ------ Reading model value used as a priori (for EOP only)
!
         IF ( I_TYP .EQ. IXPL_TYP   .OR.   I_TYP .EQ. IYPL_TYP  .OR. &
     &        I_TYP .EQ. IUT1_TYP  ) THEN
              IBG = 67  + 39*(ICOL-1)
              IEN = IBG + 12
!
              CALL DFOR_MEN ( BUF(J2)(IBG:IEN), MODU_ARR(L_PTS), IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 9236, IUER, 'GETSTA_PTS', 'Error reading '// &
     &                 'a priori values of of the ploted parameter: '// &
     &                  BUF(J2)(IBG:IEN) )
                   RETURN
              END IF
!
! ----------- Compute a shift
!
              IF ( L_PTS .EQ. 1 ) MODU_SHIFT = ( VAL_ARR(1) - MODU_ARR(1) )
!
! ----------- Compute another model value used for comparison only
!
              CALL RETURN_HF_EOP ( JD, DUT1, DX, DY, INT2(1) )
              IF ( I_TYP .EQ. IXPL_TYP ) MODC_ARR(L_PTS) = DX
              IF ( I_TYP .EQ. IYPL_TYP ) MODC_ARR(L_PTS) = DY
              IF ( I_TYP .EQ. IUT1_TYP ) MODC_ARR(L_PTS) = DUT1
              IF ( L_PTS .EQ. 1 ) THEN
                   MODU_SHIFT = ( VAL_ARR(1) - MODU_ARR(1) )
                   MODC_SHIFT = ( VAL_ARR(1) - MODC_ARR(1) )
              END IF
!
! ----------- Apply shifts
!
              MODU_ARR(L_PTS) = MODU_ARR(L_PTS) + MODU_SHIFT
              MODC_ARR(L_PTS) = MODC_ARR(L_PTS) + MODC_SHIFT
           ELSE
!
! ----------- Zero for non-EOP plotting types
!
              MODU_ARR(L_PTS) = 0.0
              MODC_ARR(L_PTS) = 0.0
         END IF
 420  CONTINUE
      CALL ERR_LOG ( 0, IUER )
!
      RETURN
      END  !#!  GETSTA_PTS  #!#
