        SUBROUTINE TRAN ( IC, ISTR, OSTR )
! ************************************************************************
! *                                                                      *
! *     Routine TRAN recodes the input string ISTR to the output string  *
! *   OSTR using the coding table which is specified by an argument IC.  *
! *   Only parameters IC=11, IC=12 and IC=13 are interesting for         *
! *   non-Russian speakers.                                              *
! *                                                                      *
! * _______________________ Input parameters : ________________________  *
! *                                                                      *
! *     IC   ( INTEGER*4 )  --  Recoding code:                           *
! *          IC=11 : all symbols are characters become the characters    *
! *                  of the upper register.                              *
! *          IC=12 : all symbols are characters become the characters    *
! *                  of the lower register.                              *
! *          IC=13 : unprintable characters are replaced with blanks.    *
! *                                                                      *
! *          other tables:                                               *
! *                                                                      *
! *          IC=1  :  DKOI ( Russian extension of EBSDIC ) --> ASCII_N1U *
! *          IC=2  :  ASCII_N1U --> DKOI ( Russian extension of EBSDIC ) *
! *          IC=3  :  ASCII_NAP ( LAT/RYS )  ---> ASCII_N1U              *
! *          IC=4  :  ASCII 7 bit Russified  ---> ASCII_N1U              *
! *          IC=5  :  ASCII_N1U  --->  ASCII_NAP (  LAT/RUS )            *
! *          IC=6  :  ASCII 8 bit RUS ( LAT+RUS ) ---> ASCII_N1U         *
! *          IC=7  :  ASCII_N1U  ---> ASCII 8 bit Russified ( LAT+RUS ). *
! *          IC=8  :  ASCII Alternative coding  --->  ASCII_N1U          *
! *          IC=9  :  ASCII_N1U  --->  ASCII Alternative coding.         *
! *          IC=10 :  ASCII_N1U  small Rus. let --->  large rus. let.    *
! *          IC=11 :  any case --->  upper case.                         *
! *          IC=12 :  any case --->  lower case.                         *
! *          IC=13 :  unprintable characters are replaced with blanks.   *
! *          IC=14 :  VAX LAT/BRITISH  --->  ASCII_N1U  RUS.             *
! *          IC=15 :  ASCII_N1U  --->  Printer CM 6361.                  *
! *          IC=16 :  Auxilliary table BASE64  ---> ASCII-8              *
! *          IC=17 :  ASCII-8 ---> Auxilliary table BASE64               *
! *                                                                      *
! *     ISTR ( CHARACTER )  --  Input string                             *
! *                                                                      *
! * ________________________ Ouput parameters: _________________________ *
! *                                                                      *
! *     OSTR ( CHARACTER )  --  Output string.                           *
! *                                                                      *
! *     NB: CALL TRAN ( IC, ISTR, ISTR ) is acceptable.                  *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ###  04-SEP-1991     TRAN    v3.13 (d) L. Petrov  04-MAY-2004  ###  *
! *                                                                      *
! ************************************************************************
        CHARACTER ISTR*(*), OSTR*(*)
        PARAMETER ( IC_LIM=16 )
        INTEGER*1 TABL(0:255,IC_LIM)
!
! ----- Recoding table  DKOI ( Russian extension of EBSDIC ) --> ASCII_N1U
!                       ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,1),NN=0,255 ) / &
     &     0,   1,   2,   3,  92,   9,  92, 127,  92,  92, &   !   0  -    9
     &    92,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    92,  92,   8,  92,  24,  25,  92,  92,  28,  29, &   !  20  -   29
     &    30,  31,  92,  92,  92,  92,  92,  10,  23,  27, &   !  30  -   39
     &    92,  92,  92,  92,  92,   5,   6,   7,  92,  92, &   !  40  -   49
     &    22,  92,  92,  92,  92,   4,  92,  92,  92,  92, &   !  50  -   59
     &    20,  21,  92,  26,  32,  92,  92,  92,  92,  92, &   !  60  -   69
     &    92,  92,  92,  92,  91,  46,  60,  40,  43,  33, &   !  70  -   79
     &    38,  92,  92,  92,  92,  92,  92,  92,  92,  92, &   !  80  -   89
     &    93,  36,  42,  41,  59,  94,  45,  47,  92,  92, &   !  90  -   99
!
     &    92,  92,  92,  92,  92,  92, 124,  44,  37,  95, &   ! 100  -  109
     &    62,  63,  92,  92,  92,  92,  92,  92, -18, -48, &   ! 110  -  119
     &   -47,  96,  58,  35,  64,  39,  61,  34, -26,  97, &   ! 120  -  129
     &    98,  99, 100, 101, 102, 103, 104, 105, -44, -43, &   ! 130  -  139
     &   -28, -45, -27, -40, -39, 106, 107, 108, 109, 110, &   ! 140  -  149
     &   111, 112, 113, 114, -38, -37, -36, -35, -34, -33, &   ! 150  -  159
     &   -17, 126, 115, 116, 117, 118, 119, 120, 121, 122, &   ! 160  -  169
     &   -32, -31, -30, -29, -42, -46, -20, -21, -41, -24, &   ! 170  -  179
     &   -19, -23, -25, -22, -50, -80, -79, -58, -76, -75, &   ! 180  -  189
     &   -60, -77, 123,  65,  66,  67,  68,  69,  70,  71, &   ! 190  -  199
!
     &    72,  73, -59, -72, -71, -70, -69, -68, 125,  74, &   ! 200  -  209
     &    75,  76,  77,  78,  79,  80,  81,  82, -67, -66, &   ! 210  -  219
     &   -65, -49, -64, -63,  92,  92,  83,  84,  85,  86, &   ! 220  -  229
     &    87,  88,  89,  90, -62, -61, -74, -78, -52, -53, &   ! 230  -  239
     &    48,  49,  50,  51,  52,  53,  54,  55,  56,  57, &   ! 240  -  249
     &   -73, -56, -51, -55, -57,  -1                    /  ! 250  -  255
!
! ----- ðåòåëïäéòï÷ïþîáñ ôáâìéãá  ASCII_î1õ  ---->  äëïé
!                                 ~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,2),NN=0,255 ) / &
     &     0,   1,   2,   3,  55,  45,  46,  47,  22,   5, &   !   0  -    9
     &    37,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    60,  61,  50,  38,  24,  25,  63,  39,  28,  29, &   !  20  -   29
     &    30,  31,  64,  79, 127, 123,  91, 108,  80, 125, &   !  30  -   39
     &    77,  93,  92,  78, 107,  96,  75,  97, -16, -15, &   !  40  -   49
     &   -14, -13, -12, -11, -10,  -9,  -8,  -7, 122,  94, &   !  50  -   59
     &    76, 126, 110, 111, 124, -63, -62, -61, -60, -59, &   !  60  -   69
     &   -58, -57, -56, -55, -47, -46, -45, -44, -43, -42, &   !  70  -   79
     &   -41, -40, -39, -30, -29, -28, -27, -26, -25, -24, &   !  80  -   89
     &   -23,  74, -32,  90,  95, 109, 121,-127,-126,-125, &   !  90  -   99
!
     &  -124,-123,-122,-121,-120,-119,-111,-110,-109,-108, &   ! 100  -  109
     &  -107,-106,-105,-104,-103, -94, -93, -92, -91, -90, &   ! 110  -  119
     &   -89, -88, -87, -64, 106, -48, -95,   7, -32, -32, &   ! 120  -  129
     &   -32, -32, -32, -32, -32, -32, -32, -32, -32, -32, &   ! 130  -  139
     &   -32, -32, -32, -32, -32, -32, -32, -32, -32, -32, &   ! 140  -  149
     &   -32, -32, -32, -32, -32, -32, -32, -32, -32, -32, &   ! 150  -  159
     &   -32, -32, -32, -32, -32, -32, -32, -32, -32, -32, &   ! 160  -  169
     &   -32, -32, -32, -32, -32, -32, -71, -70, -19, -65, &   ! 170  -  179
     &   -68, -67, -20,  -6, -53, -52, -51, -50, -49, -38, &   ! 180  -  189
     &   -37, -36, -34, -33, -22, -21, -66, -54, -69,  -2, &   ! 190  -  199
!
     &    -5,  -3, -32, -17, -18,  -4, -72, -35, 119, 120, &   ! 200  -  209
     &   -81,-115,-118,-117, -82, -78,-113,-112,-102,-101, &   ! 210  -  219
     &  -100, -99, -98, -97, -86, -85, -84, -83,-116,-114, &   ! 220  -  229
     &  -128, -74, -77, -75, -73, -79, -80, -76, 118, -96, &   ! 230  -  239
     &   -67,-117, -32, -32, -32, -32, -32, -32, -32, -32, &   ! 240  -  249
     &   -32, -32, -32, -32, -32,  -1                    /  ! 250  -  255
!
! ----- ðåòåëïäéòï÷ïþîáñ ôáâìéãá ASCII_îáð ( LAT/òõó )  ---->  ASCII_î1õ
!                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,3),NN=0,255 ) / &
     &     0,   1,   2,   3,   4,   5,   6,   7,   8,   9, &   !   0  -    9
     &    10,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    20,  21,  22,  23,  24,  25,  26,  27,  28,  29, &   !  20  -   29
     &    30,  31,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  65,  66,  67,  68,  69, &   !  60  -   69
     &    70,  71,  72,  73,  74,  75,  76,  77,  78,  79, &   !  70  -   79
     &    80,  81,  82,  83,  84,  85,  86,  87,  88,  89, &   !  80  -   89
     &    90,  91,  92,  93,  94,  95, -50, -80, -79, -58, &   !  90  -   99
!
     &   -76, -75, -60, -77, -59, -72, -71, -70, -69, -68, &   ! 100  -  109
     &   -67, -66, -65, -49, -64, -63, -62, -61, -74, -78, &   ! 110  -  119
     &   -52, -53, -73, -56, -51, -55, -57, 127,   1,   0, &   ! 120  -  129
     &    14,   1,  75, -71, -12, 127,   0,   0,   0,   0, &   ! 130  -  139
     &     0,   0,   0,   0,   0,   0,  14,   1,   0,   0, &   ! 140  -  149
     &     0,   0,   0,   0,  14,   1,   0,   0,   0,   0, &   ! 150  -  159
     &     1,   0,   0,   0, -52,  13,   0,   0,   1,   0, &   ! 160  -  169
     &    14,   1,   0,   0,   0,   0,   2,   0,   0,   0, &   ! 170  -  179
     &   -28,  13,   0,   0, -56,  13,   0,   0, -65, -49, &   ! 180  -  189
     &   -64, -63, -62, -61, -74, -78, -52, -53, -73, -56, &   ! 190  -  199
!
     &   -51, -55, -57,  32,   0,   0,  14,   1,   0,   0, &   ! 200  -  209
     &     0,   0,   0,   0,   0,   0,   1,   0,  14,   1, &   ! 210  -  219
     &     1,   5,   0,   0,   0,   0,   0,   0,   0,   0, &   ! 220  -  229
     &     0,   0,   0,   0,   0,   0,   0,   0,   0,   0, &   ! 230  -  239
     &     0,   0,   0,   0,   0,   0,   0,   0,   0,   0, &   ! 240  -  249
     &     0,   0,   0,   0,   0,   0                    /  ! 250  -  255
!
! ----- ðåòåëïäéòï÷ïþîáñ ôáâìéãá ASCII 7 âéô òõóéæéãéòï÷áîîùê --->  ASCII_î1õ
!                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,4),NN=0,255 ) / &
     &     0,   1,   2,   3,   4,   5,   6,   7,   8,   9, &   !   0  -    9
     &    10,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    20,  21,  22,  23,  24,  25,  26,  27,  28,  29, &   !  20  -   29
     &    30,  31,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  65,  66,  67,  68,  69, &   !  60  -   69
     &    70,  71,  72,  73,  74,  75,  76,  77,  78,  79, &   !  70  -   79
     &    80,  81,  82,  83,  84,  85,  86,  87,  88,  89, &   !  80  -   89
     &    90,  91,  92,  93,  94,  95,  96,  97,  98,  99, &   !  90  -   99
!
     &   100, 101, 102, 103, 104, 105, 106, 107, 108, 109, &   ! 100  -  109
     &   110, 111, 112, 113, 114, 115, 116, 117, 118, 119, &   ! 110  -  119
     &   120, 121, 122, 123, 124, 125, 126, 127,-128,-127, &   ! 120  -  129
     &  -126,-125,-124,-123,-122,-121,-120,-119,-118,-117, &   ! 130  -  139
     &  -116,-115,-114,-113,-112,-111,-110,-109,-108,-107, &   ! 140  -  149
     &  -106,-105,-104,-103,-102,-101,-100, -99, -98, -97, &   ! 150  -  159
     &   -96, -95, -94, -93, -92, -91, -90, -89, -88, -87, &   ! 160  -  169
     &   -86, -85, -84, -83, -82, -81, -80, -79, -78, -77, &   ! 170  -  179
     &   -76, -75, -74, -73, -72, -71, -70, -69, -68, -67, &   ! 180  -  189
     &   -66, -65, -64, -63, -62, -61, -60, -59, -58, -57, &   ! 190  -  199
!
     &   -56, -55, -54, -53, -52, -51, -50, -49, -48, -47, &   ! 200  -  209
     &   -46, -45, -44, -43, -42, -41, -40, -39, -38, -37, &   ! 210  -  219
     &   -36, -35, -34, -33, -50, -31, -79, -58, -76, -27, &   ! 220  -  229
     &   -60, -77, -24, -72, -71, -21, -69, -19, -18, -17, &   ! 230  -  239
     &   -65, -49, -64, -63, -62, -61, -74, -78, -52, -53, &   ! 240  -  249
     &   -73, -56, -51, -55, -57,  32                    /  ! 250  -  255
!
! ----- ðåòåëïäéòï÷ïþîáñ ôáâìéãá ASCII_î1õ  --->  ASCII_îáð LAT/òõó
!                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,5),NN=0,255 ) / &
     &     0,   1,   2,   3,   4,   5,   6,   7,   8,   9, &   !   0  -    9
     &    10,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    20,  21,  22,  23,  24,  25,  26,  27,  28,  29, &   !  20  -   29
     &    30,  31,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  65,  66,  67,  68,  69, &   !  60  -   69
     &    70,  71,  72,  73,  74,  75,  76,  77,  78,  79, &   !  70  -   79
     &    80,  81,  82,  83,  84,  85,  86,  87,  88,  89, &   !  80  -   89
     &    90,  91,  92,  93,  94,  95,  96,  65,  66,  67, &   !  90  -   99
!
     &    68,  69,  70,  71,  72,  73,  74,  75,  76,  77, &   ! 100  -  109
     &    78,  79,  80,  81,  82,  83,  84,  85,  86,  87, &   ! 110  -  119
     &    88,  89,  90,  91,  73,  93,  94, 127,   0,   1, &   ! 120  -  129
     &     2,   3,   4,   5,   6,   7,   8,   9,  10,  11, &   ! 130  -  139
     &    12,  13,  14,  15,  16,  17,  18,  19,  20,  21, &   ! 140  -  149
     &    22,  23,  24,  25,  26,  27,  28,  29,  30,  31, &   ! 150  -  159
     &    32,  33,  34,  35,  36,  37,  38,  39,  40,  41, &   ! 160  -  169
     &    42,  43,  44,  45,  46,  47,  97,  98, 119, 103, &   ! 170  -  179
     &   100, 101, 118, 122, 105, 106, 107, 108, 109, 110, &   ! 180  -  189
     &   111, 112, 114, 115, 116, 117, 102, 104,  99, 126, &   ! 190  -  199
!
     &   123, 125,  39, 121, 120, 124,  96, 113,  97,  98, &   ! 200  -  209
     &   119, 103, 100, 101, 118, 122, 105, 106, 107, 108, &   ! 210  -  219
     &   109, 110, 111, 112, 114, 115, 116, 117, 102, 104, &   ! 220  -  229
     &    99, 126, 123, 125,  39, 121, 120, 124,  96, 113, &   ! 230  -  239
     &   101, 101,  32,  32,  32,  32,  32,  32,  32,  32, &   ! 240  -  249
     &    32,  32,  32,  32,  32,  32                    /  ! 250  -  255
!
! ----- ðåòåëïäéòï÷ïþîáñ ôáâìéãá ASCII 8 âéô òõó. ( ìáô+òõó) ---> ASCII_î1õ
!                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,6),NN=0,255 ) / &
     &     0,   1,   2,   3,   4,   5,   6,   7,   8,   9, &   !   0  -    9
     &    10,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    20,  21,  22,  23,  24,  25,  26,  27,  28,  29, &   !  20  -   29
     &    30,  31,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  65,  66,  67,  68,  69, &   !  60  -   69
     &    70,  71,  72,  73,  74,  75,  76,  77,  78,  79, &   !  70  -   79
     &    80,  81,  82,  83,  84,  85,  86,  87,  88,  89, &   !  80  -   89
     &    90,  91,  92,  93,  94,  95,  96,  97,  98,  99, &   !  90  -   99
!
     &   100, 101, 102, 103, 104, 105, 106, 107, 108, 109, &   ! 100  -  109
     &   110, 111, 112, 113, 114, 115, 116, 117, 118, 119, &   ! 110  -  119
     &   120, 121, 122, 123, 124, 125, 126, 127,-128,-127, &   ! 120  -  129
     &  -126,-125,-124,-123,-122,-121,-120,-119,-118,-117, &   ! 130  -  139
     &  -116,-115,-114,-113,-112,-111,-110,-109,-108,-107, &   ! 130  -  139
     &  -106,-105,-104,-103,-102,-101,-100, -99, -98, -97, &   ! 150  -  159
     &   -96,  33,  34,  35,  36,  37,  38,  39,  40,  41, &   ! 160  -  169
     &    42,  43,  44,  45,  46,  47,  48,  49,  50,  51, &   ! 170  -  179
     &    52,  53,  54,  55,  56,  57,  58,  59,  60,  61, &   ! 180  -  189
     &    62,  63, -18, -48, -47, -26, -44, -43, -28, -45, &   ! 190  -  199
!
     &   -27, -40, -39, -38, -37, -36, -35, -34, -33, -17, &   ! 200  -  209
     &   -32, -31, -30, -29, -42, -46, -20, -21, -41, -24, &   ! 210  -  219
     &   -19, -23, -25, -22, -50, -80, -79, -58, -76, -75, &   ! 220  -  229
     &   -60, -77, -59, -72, -71, -70, -69, -68, -67, -66, &   ! 230  -  239
     &   -65, -49, -64, -63, -62, -61, -74, -78, -52, -53, &   ! 240  -  249
     &   -73, -56, -51, -55, -57, -54                    /  ! 250  -  255
!
! ----- ðåòåëïäéòï÷ïþîáñ ôáâìéãá ASCII_î1õ  ---> ASCII 8 âéô òõó. ( ìáô+òõó)
!                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,7),NN=0,255 ) / &
     &     0,   1,   2,   3,   4,   5,   6,   7,   8,   9, &   !   0  -    9
     &    10,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    20,  21,  22,  23,  24,  25,  26,  27,  28,  29, &   !  20  -   29
     &    30,  31,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  65,  66,  67,  68,  69, &   !  60  -   69
     &    70,  71,  72,  73,  74,  75,  76,  77,  78,  79, &   !  70  -   79
     &    80,  81,  82,  83,  84,  85,  86,  87,  88,  89, &   !  80  -   89
     &    90,  91,  92,  93,  94,  95,  96,  97,  98,  99, &   !  90  -   99
!
     &   100, 101, 102, 103, 104, 105, 106, 107, 108, 109, &   ! 100  -  109
     &   110, 111, 112, 113, 114, 115, 116, 117, 118, 119, &   ! 110  -  119
     &   120, 121, 122, 123, 124, 125, 126, 127,-128,-127, &   ! 120  -  129
     &  -126,-125,-124,-123,-122,-121,-120,-119,-118,-117, &   ! 130  -  139
     &  -116,-115,-114,-113,-112,-111,-110,-109,-108,-107, &   ! 140  -  149
     &  -106,-105,-104,-103,-102,-101,-100, -99, -98, -97, &   ! 150  -  159
     &   -96, -27, -94, -93, -92, -91, -90, -89, -88, -87, &   ! 160  -  169
     &   -86, -85, -84, -83, -82, -81, -31, -30,  -9, -25, &   ! 170  -  179
     &   -28, -27, -10,  -6, -23, -22, -21, -20, -19, -18, &   ! 180  -  189
     &   -17, -16, -14, -13, -12, -11, -26, -24, -29,  -2, &   ! 190  -  199
!
     &    -5,  -3,  -1,  -7,  -8,  -4, -32, -15, -63, -62, &   ! 200  -  209
     &   -41, -57, -60, -59, -42, -38, -55, -54, -53, -52, &   ! 210  -  219
     &   -51, -50, -49, -48, -46, -45, -44, -43, -58, -56, &   ! 220  -  229
     &   -61, -34, -37, -35, -33, -39, -40, -36, -64, -47, &   ! 230  -  239
     &   -16, -59, -14, -13, -12, -11, -10,  -9,  -8,  -7, &   ! 240  -  249
     &    -6,  -5,  -4,  -3,  -2,  -1                    /  ! 250  -  255
!
!
! ----- ðåòåëïäéòï÷ïþîáñ ôáâìéãá ASCII áìøôåòîáôé÷î. ëïäéòï÷ëá ---> ASCII_î1õ
!                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        DATA ( TABL(NN,8),NN=0,255 ) / &
     &     0,   1,   2,   3,   4,   5,   6,   7,   8,   9, &   !   0  -    9
     &    10,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    20,  21,  22,  23,  24,  25,  26,  27,  28,  29, &   !  20  -   29
     &    30,  31,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  65,  66,  67,  68,  69, &   !  60  -   69
     &    70,  71,  72,  73,  74,  75,  76,  77,  78,  79, &   !  70  -   79
     &    80,  81,  82,  83,  84,  85,  86,  87,  88,  89, &   !  80  -   89
     &    90,  91,  92,  93,  94,  95,  96,  97,  98,  99, &   !  90  -   99
!
     &   100, 101, 102, 103, 104, 105, 106, 107, 108, 109, &   ! 100  -  109
     &   110, 111, 112, 113, 114, 115, 116, 117, 118, 119, &   ! 110  -  119
     &   120, 121, 122, 123, 124, 125, 126, 127, -80, -79, &   ! 120  -  129
     &   -78, -77, -76, -75, -74, -73, -72, -71, -70, -69, &   ! 130  -  139
     &   -68, -67, -66, -65, -64, -63, -62, -61, -60, -59, &   ! 140  -  149
     &   -58, -57, -56, -55, -54, -53, -52, -51, -50, -49, &   ! 150  -  159
     &   -48, -47, -46, -45, -44, -43, -42, -41, -40, -39, &   ! 160  -  169
     &   -38, -37, -36, -35, -34, -33,  42,  42,  42, 124, &   ! 170  -  179
     &   -33, 124, 124, 124, 124, 124, 124, 124, 124, 124, &   ! 180  -  189
     &   124, 124, 124, 124, 124, 124,  45, 124,  45, 124, &   ! 190  -  199
!
     &   124, 124, 124,  95, 124,  95, 124,  45,  45,  95, &   ! 200  -  209
     &    95, 124, 124,  45,  45, 124, 124,  45,  95,  42, &   ! 210  -  219
     &    42,  42,  42,  42, -32, -31, -30, -29, -28, -27, &   ! 220  -  229
     &   -26, -25, -24, -23, -22, -21, -20, -19, -18, -17, &   ! 230  -  239
     &   -95, -15,  47,  92,  47,  92,  45,  45, 124, 124, &   ! 240  -  249
     &    45,  45,  78,  36,  46,   0                    /  ! 250  -  255
!
!
! ----- ðåòåëïäéòï÷ïþîáñ ôáâìéãá ASCII_î1õ ---> ASCII áìøôåòîáôé÷î. ëïäéòï÷ëá
!                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        DATA ( TABL(NN,9),NN=0,255 ) / &
     &     0,   1,   2,   3,   4,   5,   6,   7,   8,   9, &   !   0  -    9
     &    10,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    20,  21,  22,  23,  24,  25,  26,  27,  28,  29, &   !  20  -   29
     &    30,  31,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  65,  66,  67,  68,  69, &   !  60  -   69
     &    70,  71,  72,  73,  74,  75,  76,  77,  78,  79, &   !  70  -   79
     &    80,  81,  82,  83,  84,  85,  86,  87,  88,  89, &   !  80  -   89
     &    90,  91,  92,  93,  94,  95,  96,  97,  98,  99, &   !  90  -   99
!
     &   100, 101, 102, 103, 104, 105, 106, 107, 108, 109, &   ! 100  -  109
     &   110, 111, 112, 113, 114, 115, 116, 117, 118, 119, &   ! 110  -  119
     &   120, 121, 122, 123, 124, 125, 126, 127,   0,   0, &   ! 120  -  129
     &     0,   0,   0,   0,   0,   0,   0,   0,   0,   0, &   ! 130  -  139
     &     0,   0,   0,   0,   0,   0,   0,   0,   0,   0, &   ! 140  -  149
     &     0,   0,   0,   0,   0,   0,   0,   0,   0,   0, &   ! 150  -  159
     &     0, -16,   0,   0,   0,   0,   0,   0,   0,   0, &   ! 160  -  169
     &     0,   0,   0,   0,   0,   0,-128,-127,-126,-125, &   ! 170  -  179
     &  -124,-123,-122,-121,-120,-119,-118,-117,-116,-115, &   ! 180  -  189
     &  -114,-113,-112,-111,-110,-109,-108,-107,-106,-105, &   ! 190  -  199
!
     &  -104,-103,-102,-101,-100, -99, -98, -97, -96, -95, &   ! 200  -  209
     &   -94, -93, -92, -91, -90, -89, -88, -87, -86, -85, &   ! 210  -  219
     &   -84, -83, -82, -81, -32, -31, -30, -29, -28, -27, &   ! 220  -  229
     &   -26, -25, -24, -23, -22, -21, -20, -19, -18, -17, &   ! 230  -  239
     &   -16, -15,   0,   0,   0,   0,   0,   0,   0,   0, &   ! 240  -  249
     &     0,   0,   0,   0,   0,   0                    /  ! 250  -  255
!
! ----- ðåòåëïäéòï÷ïþîáñ ôáâìéãá ASCII_î1õ  íáì. òõó. --->  âïì. òõó.
!                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,10),NN=0,255 ) / &
     &     0,   1,   2,   3,   4,   5,   6,   7,   8,   9, &   !   0  -    9
     &    10,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    20,  21,  22,  23,  24,  25,  26,  27,  28,  29, &   !  20  -   29
     &    30,  31,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  65,  66,  67,  68,  69, &   !  60  -   69
     &    70,  71,  72,  73,  74,  75,  76,  77,  78,  79, &   !  70  -   79
     &    80,  81,  82,  83,  84,  85,  86,  87,  88,  89, &   !  80  -   89
     &    90,  91,  92,  93,  94,  95,  96,  97,  98,  99, &   !  90  -   99
!
     &   100, 101, 102, 103, 104, 105, 106, 107, 108, 109, &   ! 100  -  109
     &   110, 111, 112, 113, 114, 115, 116, 117, 118, 119, &   ! 110  -  119
     &   120, 121, 122, 123, 124, 125, 126, 127,-128,-127, &   ! 120  -  129
     &  -126,-125,-124,-123,-122,-121,-120,-119,-118,-117, &   ! 130  -  139
     &  -116,-115,-114,-113,-112,-111,-110,-109,-108,-107, &   ! 130  -  139
     &  -106,-105,-104,-103,-102,-101,-100, -99, -98, -97, &   ! 150  -  159
     &   -96, -75, -94, -93, -92, -91, -90, -89, -88, -87, &   ! 160  -  169
     &   -86, -85, -84, -83, -82, -81, -80, -79, -78, -77, &   ! 170  -  179
     &   -76, -75, -74, -73, -72, -71, -70, -69, -68, -67, &   ! 180  -  189
     &   -66, -65, -64, -63, -62, -61, -60, -59, -58, -57, &   ! 190  -  199
!
     &   -56, -55, -54, -53, -52, -51, -50, -49, -80, -79, &   ! 200  -  209
     &   -78, -77, -76, -75, -74, -73, -72, -71, -70, -69, &   ! 210  -  219
     &   -68, -67, -66, -65, -64, -63, -62, -61, -60, -59, &   ! 220  -  229
     &   -58, -57, -56, -55, -54, -53, -52, -51, -50, -49, &   ! 230  -  239
     &   -95, -75, -14, -13, -12, -11, -10,  -9,  -8,  -7, &   ! 240  -  249
     &    -6,  -5,  -4,  -3,  -2,  -1                    /  ! 250  -  255
!
! ----- Recoding table: any case --> upper case
!                       ~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,11),NN=0,255 ) / &
     &     0,   1,   2,   3,   4,   5,   6,   7,   8,   9, &   !   0  -    9
     &    10,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    20,  21,  22,  23,  24,  25,  26,  27,  28,  29, &   !  20  -   29
     &    30,  31,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  65,  66,  67,  68,  69, &   !  60  -   69
     &    70,  71,  72,  73,  74,  75,  76,  77,  78,  79, &   !  70  -   79
     &    80,  81,  82,  83,  84,  85,  86,  87,  88,  89, &   !  80  -   89
     &    90,  91,  92,  93,  94,  95,  96,  65,  66,  67, &   !  90  -   99
!
     &    68,  69,  70,  71,  72,  73,  74,  75,  76,  77, &   ! 100  -  109
     &    78,  79,  80,  81,  82,  83,  84,  85,  86,  87, &   ! 100  -  109
     &    88,  89,  90, 123, 124, 125, 126, 127,-128,-127, &   ! 120  -  129
     &  -126,-125,-124,-123,-122,-121,-120,-119,-118,-117, &   ! 130  -  139
     &  -116,-115,-114,-113,-112,-111,-110,-109,-108,-107, &   ! 130  -  139
     &  -106,-105,-104,-103,-102,-101,-100, -99, -98, -97, &   ! 150  -  159
     &   -96, -75, -94, -93, -92, -91, -90, -89, -88, -87, &   ! 160  -  169
     &   -86, -85, -84, -83, -82, -81, -80, -79, -78, -77, &   ! 170  -  179
     &   -76, -75, -74, -73, -72, -71, -70, -69, -68, -67, &   ! 180  -  189
     &   -66, -65, -64, -63, -62, -61, -60, -59, -58, -57, &   ! 190  -  199
!
     &   -56, -55, -54, -53, -52, -51, -50, -49, -80, -79, &   ! 200  -  209
     &   -78, -77, -76, -75, -74, -73, -72, -71, -70, -69, &   ! 210  -  219
     &   -68, -67, -66, -65, -64, -63, -62, -61, -60, -59, &   ! 220  -  229
     &   -58, -57, -56, -55, -54, -53, -52, -51, -50, -49, &   ! 230  -  239
     &   -95, -75, -14, -13, -12, -11, -10,  -9,  -8,  -7, &   ! 240  -  249
     &    -6,  -5,  -4,  -3,  -2,  -1                    /  ! 250  -  255
!
!
! ----- Recoding table: any case --> lower case.
!                       ~~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,12),NN=0,255 ) / &
     &     0,   1,   2,   3,   4,   5,   6,   7,   8,   9, &   !   0  -    9
     &    10,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    20,  21,  22,  23,  24,  25,  26,  27,  28,  29, &   !  20  -   29
     &    30,  31,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  97,  98,  99, 100, 101, &   !  60  -   69
     &   102, 103, 104, 105, 106, 107, 108, 109, 110, 111, &   !  70  -   79
     &   112, 113, 114, 115, 116, 117, 118, 119, 120, 121, &   !  80  -   89
     &   122,  91,  92,  93,  94,  95,  96,  97,  98,  99, &   !  90  -   99
!
     &   100, 101, 102, 103, 104, 105, 106, 107, 108, 109, &   ! 100  -  109
     &   110, 111, 112, 113, 114, 115, 116, 117, 118, 119, &   ! 110  -  119
     &   120, 121, 122, 123, 124, 125, 126, 127,-128,-127, &   ! 120  -  129
     &  -126,-125,-124,-123,-122,-121,-120,-119,-118,-117, &   ! 130  -  139
     &  -116,-115,-114,-113,-112,-111,-110,-109,-108,-107, &   ! 130  -  139
     &  -106,-105,-104,-103,-102,-101,-100, -99, -98, -97, &   ! 150  -  159
     &   -96, -15, -94, -93, -92, -91, -90, -89, -88, -87, &   ! 160  -  169
     &   -86, -85, -84, -83, -82, -81, -48, -47, -46, -45, &   ! 170  -  179
     &   -44, -43, -42, -41, -40, -39, -38, -37, -36, -35, &   ! 180  -  189
     &   -34, -33, -32, -31, -30, -29, -28, -27, -26, -25, &   ! 190  -  199
!
     &   -24, -23, -22, -21, -20, -19, -18, -17, -48, -47, &   ! 200  -  209
     &   -46, -45, -44, -43, -42, -41, -40, -39, -38, -37, &   ! 210  -  219
     &   -36, -35, -34, -33, -32, -31, -30, -29, -28, -27, &   ! 220  -  229
     &   -26, -25, -24, -23, -22, -21, -20, -19, -18, -17, &   ! 230  -  239
     &   -15, -15, -14, -13, -12, -11, -10,  -9,  -8,  -7, &   ! 240  -  249
     &    -6,  -5,  -4,  -3,  -2,  -1                    /  ! 250  -  255
!
!
! ----- Recording table: Replacement of unprintable characters with blanks
!                        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,13),NN=0,255 ) / &
     &    32,  32,  32,  32,  32,  32,  32,  32,  32,   9, &   !   0  -    9
     &    32,  32,  32,  32,  32,  32,  32,  32,  32,  32, &   !  10  -   19
     &    32,  32,  32,  32,  32,  32,  32,  32,  32,  32, &   !  20  -   29
     &    32,  32,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  65,  66,  67,  68,  69, &   !  60  -   69
     &    70,  71,  72,  73,  74,  75,  76,  77,  78,  79, &   !  70  -   79
     &    80,  81,  82,  83,  84,  85,  86,  87,  88,  89, &   !  80  -   89
     &    90,  91,  92,  93,  94,  95,  96,  97,  98,  99, &   !  90  -   99
!
     &   100, 101, 102, 103, 104, 105, 106, 107, 108, 109, &   ! 100  -  109
     &   110, 111, 112, 113, 114, 115, 116, 117, 118, 119, &   ! 110  -  119
     &   120, 121, 122, 123, 124, 125, 126,  32,  32,  32, &   ! 120  -  129
     &    32,  32,  32,  32,  32,  32,  32,  32,  32,  32, &   ! 130  -  139
     &    32,  32,  32,  32,  32,  32,  32,  32,  32,  32, &   ! 140  -  149
     &    32,  32,  32,  32,  32,  32,  32,  32,  32,  32, &   ! 150  -  159
     &    32, -95,  32,  32,  32,  32,  32,  32,  32,  32, &   ! 160  -  169
     &   -86, -85, -84, -83, -82, -81, -80, -79, -78, -77, &   ! 170  -  179
     &   -76, -75, -74, -73, -72, -71, -70, -69, -68, -67, &   ! 180  -  189
     &   -66, -65, -64, -63, -62, -61, -60, -59, -58, -57, &   ! 190  -  199
!
     &   -56, -55, -54, -53, -52, -51, -50, -49, -48, -47, &   ! 200  -  209
     &   -46, -45, -44, -43, -42, -41, -40, -39, -38, -37, &   ! 210  -  219
     &   -36, -35, -34, -33, -32, -31, -30, -29, -28, -27, &   ! 220  -  229
     &   -26, -25, -24, -23, -22, -21, -20, -19, -18, -17, &   ! 230  -  239
     &   -95, -15,  32,  32,  32,  32,  32,  32,  32,  32, &   ! 240  -  249
     &    32,  32,  32,  32,  32,  32                    /  ! 250  -  255
!
! ----- ðåòåëïäéòï÷ïþîáñ ôáâìéãá VAX LAT/BRITISH  ---> ASCII_î1õ
!                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,14),NN=0,255 ) / &
     &     0,   1,   2,   3,   4,   5,   6,   7,   8,   9, &   !   0  -    9
     &    10,  11,  12,  13,  14,  15,  16,  17,  18,  19, &   !  10  -   19
     &    20,  21,  22,  23,  24,  25,  26,  27,  28,  29, &   !  20  -   29
     &    30,  31,  32,  33, -51,  35,  36,  58,  46, -19, &   !  30  -   39
     &    40,  41,  59,  43, -79,  45, -50,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57, -74, -42, &   !  50  -   59
     &   -15,  61, -95,  63,  64, -60, -72, -63, -78, -61, &   !  60  -   69
     &   -80, -65, -64, -56, -66, -69, -76, -52, -62, -55, &   !  70  -   79
     &   -73, -71, -70, -53, -75, -77, -68, -58, -57, -67, &   !  80  -   89
     &   -49, -27,  92, -22,  44,  95,  96, -28, -40, -31, &   !  90  -   99
!
     &   -46, -29, -48, -33, -32, -24, -34, -37, -44, -20, &   ! 100  -  109
     &   -30, -23, -41, -39, -38, -21, -43, -45, -36, -26, &   ! 110  -  119
     &   -25, -35, -17, -59, 124, -54, 126, 127,-128,-127, &   ! 120  -  129
     &  -126,-125,-124,-123,-122,-121,-120,-119,-118,-117, &   ! 130  -  139
     &  -116,-115,-114,-113,-112,-111,-110,-109,-108,-107, &   ! 140  -  149
     &  -106,-105,-104,-103,-102,-101,-100, -99, -98, -97, &   ! 150  -  159
     &   -96, -95, -94, -93, -92, -91, -90, -89, -88, -87, &   ! 160  -  169
     &   -86, -85, -84, -83, -82, -81, -80, -79, -78, -77, &   ! 170  -  179
     &   -76, -75, -74, -73, -72, -71, -70, -69, -68, -67, &   ! 180  -  189
     &   -66, -65, -64, -63, -62, -61, -60, -59, -58, -57, &   ! 190  -  199
!
     &   -56, -55, -54, -53, -52, -51, -50, -49, -48, -47, &   ! 200  -  209
     &   -46, -45, -44, -43, -42, -41, -40, -39, -38, -37, &   ! 210  -  219
     &   -36, -35, -34, -33, -32, -31, -30, -29, -28, -27, &   ! 220  -  229
     &   -26, -25, -24, -23, -22, -21, -20, -19, -18, -17, &   ! 230  -  239
     &   -16, -15, -14, -13, -12, -11, -10,  -9,  -8,  -7, &   ! 240  -  249
     &    -6,  -5,  -4,  -3,  -2,  -1                    /  ! 250  -  255
!
! ----- ðåòåëïäéòï÷ïþîáñ ôáâìéãá ASCII_î1õ ---> ðòéîôåò óí 6361.
!                                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!
        DATA ( TABL(NN,15),NN=0,255 ) / &
     &    32,  32,  32,  32,  32,  32,  32,  32,  32,   9, &   !   0  -    9
     &    10,  32,  12,  13,  32,  32,  32,  32,  32,  32, &   !  10  -   19
     &    32,  32,  32,  32,  32,  32,  32,  32,  32,  32, &   !  20  -   29
     &    32,  32,  32,  33,  34,  35,  36,  37,  38,  39, &   !  30  -   39
     &    40,  41,  42,  43,  44,  45,  46,  47,  48,  49, &   !  40  -   49
     &    50,  51,  52,  53,  54,  55,  56,  57,  58,  59, &   !  50  -   59
     &    60,  61,  62,  63,  64,  65,  66,  67,  68,  69, &   !  60  -   69
     &    70,  71,  72,  73,  74,  75,  76,  77,  78,  79, &   !  70  -   79
     &    80,  81,  82,  83,  84,  85,  86,  87,  88,  89, &   !  80  -   89
     &    90,  91,  92,  93,  94,  95,  96,  97,  98,  99, &   !  90  -   99
!
     &   100, 101, 102, 103, 104, 105, 106, 107, 108, 109, &   ! 100  -  109
     &   110, 111, 112, 113, 114, 115, 116, 117, 118, 119, &   ! 110  -  119
     &   120, 121, 122, 123, 124, 125, 126,  32,  32,  32, &   ! 120  -  129
     &    32,  32,  32,  32,  32,  32,  32,  32,  32,  32, &   ! 130  -  139
     &    32,  32,  32,  32,  32,  32,  32,  32,  32,  32, &   ! 140  -  149
     &    32,  32,  32,  32,  32,  32,  32,  32,  32,  32, &   ! 150  -  159
     &    32, -27,  32,  32,  32,  32,  32,  32,  32,  32, &   ! 160  -  169
     &    32,  32,  32,  32,  32,  32,  65, -30,  66, -25, &   ! 170  -  179
     &   -28,  69, -10,  -6, -23, -22,  75, -20,  77,  72, &   ! 180  -  189
     &    79, -16,  80,  67,  84, -11, -26,  88, -29,  -2, &   ! 190  -  199
!
     &    -5,  -3, -17,  -7,  -8,  -4, -32, -15,  97, -62, &   ! 200  -  209
     &   -41, -57, -60, 101, -42, -38, -55, -54, -53, -52, &   ! 210  -  219
     &   -51, -50, 111, -48, 112,  99, -44, 121, -58, 120, &   ! 220  -  229
     &   -61, -34, -37, -35, -33, -39, -40, -36, -64, -47, &   ! 230  -  239
     &   -27, -59,  32,  32,  32,  32,  32,  32,  32,  32, &   ! 240  -  249
     &    32,  32,  32,  32,  32,  32                    /  ! 250  -  256
!
! ----- ÷óðïíïçáôåìøîáñ ôáâìéãá äìñ BASE64
!
        DATA ( TABL(NN,16),NN=0,255 ) / &
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   !   0 -     9
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   !  10 -    19
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   !  20 -    29
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   !  30 -    39
     &    -1,  -1,  -1,  62,  -1,  -1,  -1,  63,  52,  53, &   !  40 -    49
     &    54,  55,  56,  57,  58,  59,  60,  61,  -1,  -1, &   !  50 -    59
     &    -1,  64,  -1,  -1,  -1,   0,   1,   2,   3,   4, &   !  60 -    69
     &     5,   6,   7,   8,   9,  10,  11,  12,  13,  14, &   !  70 -    79
     &    15,  16,  17,  18,  19,  20,  21,  22,  23,  24, &   !  80 -    89
     &    25,  -1,  -1,  -1,  -1,  -1,  -1,  26,  27,  28, &   !  90 -    99
!
     &    29,  30,  31,  32,  33,  34,  35,  36,  37,  38, &   ! 100 -   109
     &    39,  40,  41,  42,  43,  44,  45,  46,  47,  48, &   ! 110 -   119
     &    49,  50,  51,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 120 -   129
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 130 -   139
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 140 -   149
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 150 -   159
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 160 -   169
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 170 -   179
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 180 -   189
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 190 -   199
!
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 200 -   209
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 210 -   219
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 220 -   229
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 230 -   239
     &    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, &   ! 240 -   249
     &    -1,  -1,  -1,  -1,  -1,  -1                    /  ! 250 -   259
!
! ----- óïâóô÷åîîï ðåòåëïäéòï÷ëá
!
        IF ( IC.GE.2 .AND. IC.LE.IC_LIM ) THEN
             CALL MTC ( ISTR, OSTR, INT1(32), TABL(0,IC) )
          ELSE IF ( IC.EQ.1 ) THEN
             CALL MTC ( ISTR, OSTR, INT1(64), TABL(0,IC) )
          ELSE
             CALL CLRCH ( OSTR )
        END IF
        RETURN
        END  !#!  TRAN  #!#
!
! ------------------------------------------------------------------------
!
      SUBROUTINE MTC ( ISTR, OSTR, SYM, TABL )
! ************************************************************************
! *                                                                      *
! *   Auxilliary program MTC for recoding the input string ISTR to the   *
! *   output string OSTR.                                                *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ### O4-SEP-1991      MTC      v1.1 (d)  L. Petrov  04-MAY-2004 ###  *
! *                                                                      *
! ************************************************************************
      CHARACTER  ISTR*(*), OSTR*(*)
      INTEGER*1  SYM, TABL(0:255)
!
      IF ( LEN(ISTR) .LE. 0 ) RETURN
      LN = MIN( LEN(ISTR), LEN(OSTR) )
      DO 410 J1=1,LN
         IND=ICHAR(ISTR(J1:J1))
         IF ( IND .LT.0 ) IND=256+IND
         OSTR(J1:J1)=CHAR(TABL(IND))
 410  CONTINUE
!
      RETURN
      END  !#!  MTC  #!#
