#!/usr/bin/env python3
import  sys, os, subprocess, signal, time, datetime, gzip, bz2, lzma, argparse
from    vsdc_misc   import *
from    vsdc_config import *

def mjd_tim_to_time ( mjd, tim ):
    """
    Transform a pair mjd, tim into Python time
    """
    date_j2000 = datetime.datetime.strptime ( "2000", "%Y" )
    if ( mjd < 30000 or mjd > 70000 ):
         print ( "mjd %d is out of range [30000, 70000]" % mjd )
         return None
    if ( tim < -0.000001 or tim > 86400.001 ):
         print ( "tim %f is out of range [-0.001, 86400.001]" % tim )
         return None
    return ( date_j2000 + datetime.timedelta ( days=(mjd - 51544), \
                                               seconds=tim ) )

#
# ------------------------------------------------------------------------
#
def vsdc_swin_get_meta ( swin_tar_file ):
#
# --- Read the first null-terminiated string of the tar file. It should be 
# --- the name of the first file in the archive.
#
    if ( swin_tar_file[-8:] == ".tar.bz2" ):
         com = "cat " + swin_tar_file + " | bzip2 -d - | awk 'BEGIN { RS = " + '"\\0"' + " } ; {print $1} NR=1{exit}'"
    elif ( swin_tar_file[-7:] == ".tar.gz" ):
         com = "cat " + swin_tar_file + " | gzip  -d - | awk 'BEGIN { RS = " + '"\\0"' + " } ; {print $1} NR=1{exit}'"
    elif ( swin_tar_file[-4:] == ".tar" ):
         com = "cat " + swin_tar_file + " |              awk 'BEGIN { RS = " + '"\\0"' + " } ; {print $1} NR=1{exit}'"
    else:
         return ( 1, "Unsupported extension of the input file %s . " \
                    "Expected extensions: .tar.bz2 .tar.gz .tar" % swin_tar_file )

    (ret, out ) = vsdc_exe ( com )
    if ( ret != 0 ):
         return ( 1, "Error in reading the first block of the input file swin_tar_file: %s" % err[0] )

#
# --- Get the name of the meta file
#
    meta_file = out[0]

#
# --- The meta file should end at "_meta.txt" 
#
    if ( meta_file[-9:] != "_meta.txt" ):
         return ( 1, "The first file in the archive " + swin_tar_file + " is not a meta " + \
                  " file. The structure of the tar file is not supported. " + \
                  "Please use vsdc package for preparation of the " + \
                  "tar file" )
    
#
# --- Read the file with meta data from the archive
#
    if ( swin_tar_file[-8:] == ".tar.bz2" ):
         com = "cat " + swin_tar_file + " " + "| bzip2 -d - | tar --occurrence=1 -xf - " + meta_file + " --to-stdout"
    elif ( swin_tar_file[-7:] == ".tar.gz" ):
         com = "cat " + swin_tar_file + " " + "| gzip  -d - | tar --occurrence=1 -xf - " + meta_file + " --to-stdout"
    elif ( swin_tar_file[-4:] == ".tar" ):
         com = "cat " + swin_tar_file + " " + "|              tar --occurrence=1 -xf - " + meta_file + " --to-stdout"

    (ret, meta ) = vsdc_exe ( com )
    if ( ret != 0 ):
         return ( 1, meta )

    return ( 0, meta )

#
# ------------------------------------------------------------------------
#
def vsdc_swin_check ( swin_magic, swin_tar_file ):
    """
    Routine vsdc_swin_check checks a contents of the VLBI Level 1A data
    in swin format. The VLBI Level 1A is a tar file that can be compressed
    with bzip2, or gzip.

    Prerequisites:

    Since the swin_tar_file can be over 0.1 Tb, it is essential that the tar file
    is organized a special way. vsdc_swin_check expects the tar file is 
    generated by vsdc_swin_gen_tar and it will refuse to process a tar file 
    generated other way, since a check would take prohibitively expensive. 
    vsdc_swin_gen_tar generates a tar archived in a specific order:
    -- the first  file is a short ascii file with meta data;
    -- the second file is a VLBI schedule file related to this experiment in vex format;
    -- the third  file is a DiFX correlator control file;
    -- then k so-called difx input files follow sorted alphabetically;
    -- then m other files follow sorted alphabetically.

    In order to perform checks, vsdc_swin_check reads the first two files via pipe.
    That is why they have to be the first files.

    vsdc_swin_check returns a tuple of four variables on exit:

    -- message
    -- experiment name;
    -- experiment start date in Python format;
    -- experiment stop  date in Python format;

    If the file passed the check, the message is "OK". Any other message means
    the check failed and fields experiment name, experiment start, experiment 
    stop may be None.

    Last modified by Leonid Petrov  2021.05.05
    """

    exper_name = None
    date_start = None
    date_stop  = None
    vex_file   = None
    v2d_file   = None
    file_list  = []
    inp_list   = []

#
# --- Check whether the tar file exists
#
    if ( not os.path.isfile ( swin_tar_file ) ):
         return ( "Input file %s does not exist" % swin_tar_file, exper_name, date_start, date_stop )

    (ret, meta) = vsdc_swin_get_meta ( swin_tar_file )
    if ( ret !=0 ):
         return ( meta, exper_name, date_start, date_stop ) 

#
# --- Parse the meta file
#
    nl = 0
    for line in meta:
        nl = nl + 1
        if ( nl == 1 ):
#
# ---------- This the first line. Check the magic
#
             if ( line[0:25] == swin_magic ):
                  continue
             else:
                  return ( "The meta-data file has an usupported format. Expected magic: %s, but got %s" % \
                            ( swin_magic, line ), exper_name, date_start, date_stop )
        if ( len(line) < 1 ): continue
#
# ----- Extract the values of the certain keywords of the metadata file 
# ----- in order to perform checks
#
        if ( line.split()[0] == "exper_name:" ):
             exper_name = line.split()[1]
        elif ( line.split()[0] == "vex_file:" ):
             vex_file = line.split()[1]
        elif ( line.split()[0] == "v2d_file:" ):
             v2d_file = line.split()[1]
        elif ( line.split()[0] == "date_start:" ):
             date_start = datetime.datetime.strptime( line.split()[3], "%Y.%m.%d_%H:%M:%S" )
        elif ( line.split()[0] == "date_stop:" ):
             date_stop = datetime.datetime.strptime( line.split()[3], "%Y.%m.%d_%H:%M:%S" )
        elif ( line.split()[0] == "date_end:" ):
             date_stop = datetime.datetime.strptime( line.split()[3], "%Y.%m.%d_%H:%M:%S" )
        elif ( line.split()[0] == "file:" ):
             file_list.append ( line.split()[1] )
             if ( line.split()[1][-6:] == ".input" ):
                  inp_list.append ( line.split()[1] )

#
# --- Check whether all the keywords were present in the metadata file
#
    if ( not exper_name ):
         return ( "Metadata file %s of the archive file %s is missing keyword exper_name" % \
                   ( meta_file, swin_tar_file ), exper_name, date_start, date_stop )

    if ( not date_start ):
         return ( "Metadata file %s of the archive file %s is missing keyword date_start" % \
                   ( meta_file, swin_tar_file ), exper_name, date_start, date_stop )

    if ( not date_stop ):
         return ( "Metadata file %s of the archive file %s is missing keyword date_stop" % \
                   ( meta_file, swin_tar_file ), exper_name, date_start, date_stop )

    if ( not vex_file ):
         return ( "Metadata file %s of the archive file %s is missing keyword vex_file" % \
                   ( meta_file, swin_tar_file ), exper_name, date_start, date_stop )

    if ( not v2d_file ):
         return ( "Metadata file %s of the archive file %s is missing keyword v2d_file" % \
                   ( meta_file, swin_tar_file ), exper_name, date_start, date_stop )

    if ( len(inp_list) == 0 ):
         return ( "Metadata file %s of the archive file %s has no difx input files" % \
                   ( meta_file, swin_tar_file ), exper_name, date_start, date_stop )

#
# --- Check whether vex and v2d files are present in the archive file
#
    vex_file_present = None
    v2d_file_present = None
    for file in file_list:
        if ( file == vex_file ):
             vex_file_present = "yes"        
        elif ( file == v2d_file ):
             v2d_file_present = "yes"        

    if ( not vex_file_present ):
         return ( "VLBI schedule file in vex format %s is not present " \
                  "in the archive file %s" % \
                   ( vex_file, swin_tar_file ), exper_name, date_start, date_stop )

    if ( not v2d_file_present ):
         return ( "DiFX correlator control file in v2d format %s is not present " \
                  "in the archive file %s" % \
                   ( v2d_file, swin_tar_file ), exper_name, date_start, date_stop )

#
# --- Read vex file with meta data from the archive
#
    if ( swin_tar_file[-8:] == ".tar.bz2" ):
         com = "cat " + swin_tar_file + " " + "| bzip2 -d - | tar --occurrence=1 -xf - " + vex_file + " --to-stdout"
    elif ( swin_tar_file[-7:] == ".tar.gz" ):
         com = "cat " + swin_tar_file + " " + "| gzip  -d - | tar --occurrence=1 -xf - " + vex_file + " --to-stdout"
    elif ( swin_tar_file[-4:] == ".tar" ):
         com = "cat " + swin_tar_file + " " + "|              tar --occurrence=1 -xf - " + vex_file + " --to-stdout"

    (ret, vex ) = vsdc_exe ( com )
    if ( ret != 0 ):
         return ( "Error in extraction of the vex file from the archive " + \
                   swin_tar_file + " -- " + vex[0] + " during excuting command " + com, \
                   exper_name, date_start, date_stop )
#
# --- Check whether vex file corresponds to this experiment or not
#
    vex_exper_name = "undefined"
    for line in vex:
        if ( len(line.split()) > 2 ):
             if ( line.split()[0] == "exper_name" ):
                  vex_exper_name = line.split()[2].lower().replace(";","")

    if ( vex_exper_name == "undefined" ):
         return ( "vex file found in the archive %s does not have field exper_name " % \
                  ( vex_file ), exper_name, date_start, date_stop )

    if ( vex_exper_name != exper_name ):
         return ( "exper_name in metadata file is %s, but in vex file found in " \
                  "the archive %s experiment name is %s" % \
                  ( exper_name, swin_tar_file, vex_exper_name ), exper_name, date_start, date_stop )

#
# --- Now check all DiFX input files
#
    for inp_file in inp_list:
        difx_dir  = inp_file.replace(".input",".difx")
        calc_file = inp_file.replace(".input",".calc")
        im_file   = inp_file.replace(".input",".im")
        flag_file = inp_file.replace(".input",".flag")
#
        difx_dir_status  = "??"
        calc_file_status = "??" 
        im_file_status   = "??" 
        flag_file_status = "??" 
#
# ----- We need find matching files to a given input DiFX file
#
        for file in file_list:
            if ( difx_dir + "/DIFX" in file and ".b00" in file ):
                 difx_dir_status = "OK"
            elif ( file == calc_file ):
                 calc_file_status = "OK"
            elif ( file == im_file ):
                 im_file_status = "OK"
            elif ( file == flag_file ):
                 flag_file_status = "OK"
        if ( difx_dir_status != "OK" ):
             return ( "Difx input file %s does not have a corresponding " \
                      "binary visibility file in the swin archive file %s" % \
                      ( inp_file, swin_tar_file ), exper_name, date_start, date_stop )
        if ( calc_file_status != "OK" ):
             return ( "Difx input file %s does not have a corresponding calc file " \
                      "in the swin archive file %s" % ( inpfile, swin_tar_file ), exper_name, date_start, date_stop )
        if ( im_file_status != "OK" ):
             return ( "Difx input file %s does not have a corresponding im file " \
                      "in the swin archive file %s" % ( inp_file, swin_tar_file ), exper_name, date_start, date_stop )
        if ( flag_file_status != "OK" ):
             return ( "Difx input file %s does not have a corresponding flag file " \
                      "in the swin archive file %s" % ( inp_file, swin_tar_file ), exper_name, date_start, date_stop )
             
    return ( "OK", exper_name, date_start, date_stop )

#
# ------------------------------------------------------------------------
#
def vsdc_swin_gen_tar ( vsdc, dirin, v2d_file, vex_file, dirout, tmp_dir, \
                        corr_vers, compr, verb ):
    """
    Routine vsdc_swin_gen_tar generates the output tar file with the archive
    of VLBI Level 1A data in swin format.
    """

    if ( not os.path.isdir ( dirin ) ):
         print  ( "vsdc_swin_gen_tar: input directory %s does not exist" % dirin )
         return ( 1 )

    if ( verb > 1 ):
         print ( "vsds_swin: dirin=    ", dirin )
         print ( "vsds_swin: dirout=   ", dirout )
         print ( "vsds_swin: v2d_file= ", v2d_file )
         print ( "vsds_swin: vex_file= ", vex_file )

#
# --- Initialization
#
    file_list = []
    input_file_list = []
    im_file_list = []
    tel_list = []
    pol_list = []
    sou_list = []
    meta={}
    meta["exper_name"] = "??"
    meta["exper_desc"] = "??"
    meta["pi_name"]    = "??"
    meta["target_correlator"] = "??"
#
# --- Remove of trailin slash of needed
#

    if ( dirin[len(dirin)-1:len(dirin)] == "/" ):
         dirin = dirin[0:-1]
    if ( dirout[len(dirout)-1:len(dirout)] == "/" ):
         dirout = dirout[0:-1]

    if ( vex_file[0:1] != "/" ):
         vex_file = dirin + "/" + vex_file
    if ( v2d_file[0:1] != "/" ):
         v2d_file = dirin + "/" + v2d_file

    if ( not os.path.isfile ( vex_file ) ):
         print ( "Did not find vex file %s" % vex_file )
         return ( 1, None )

#
# --- Read and parse VLBI schedule file in vex format
#
    vex = read_file ( vex_file )
    for line in vex:
        if ( "exper_name" in line ):
              meta["exper_name"] = line.split()[2].lower().replace(";","")
        if ( "exper_desc" in line ):
              meta["exper_desc"] = line.replace(";","").replace("exper_description =","").replace('"','').ljust(128).strip()
        if ( "PI_name" in line ):
              meta["pi_name"]    = line.split()[2].replace(";","")
        if ( "target_correlator" in line ):
              meta["target_correlator"] = line.split()[2].replace(";","")

#
# --- Generate a name of the metadata file
#
    meta_file     = tmp_dir + "/" + meta["exper_name"] + "_meta.txt"
    tar_list_file = tmp_dir + "/" + meta["exper_name"] + "_tarlist.txt"

    tar_exc_list = ""
    if ( "VSDC_SWIN_EXCLUDE" in os.environ ):
         if ( len(os.environ["VSDC_SWIN_EXCLUDE"]) > 0 ):
              tar_exc_list = os.environ["VSDC_SWIN_EXCLUDE"].split()
         if ( verb > 1 ): 
              print ( "Got environment variable VSDC_SWIN_EXCLUDE tar_exc_list= ", tar_exc_list, flush=True )
#
# --- Walk over the swin directory file.
#
    for path, dirs, files in os.walk(dirin):
        for file in files:
#
# --------- Check all file exclude cases
#
            fl_to_exclude = False
            for tar_exc in tar_exc_list:
                if ( tar_exc != "" ):
                     if ( tar_exc in path + "/" + file ):
                          if ( verb > 3 ): print ( "EXCLUDED_file: ", path + "/" + file, flush=True )
                          fl_to_exclude = True
            if ( fl_to_exclude ):
                 continue

            if ( verb > 2 ): 
                 print ( "file= ", file, " path= ", path, flush=True )

            if ( path[len(path)-1:len(path)] == "/" ):
                 path = path[0:-1]
#
# --------- Skip some files that we do not want to have in the archieve
#
            if ( "_meta.txt" in file ): continue
            if ( "#" in file ): continue
            if ( "~" in file ): continue
            if ( not os.path.isfile ( path + "/" + file ) ): continue

            if ( ( vex_file[0:1] == "/"  and path + "/"  + file == vex_file ) or \
                   file == vex_file ):
#
# -------------- Get the name of the vex file
#
                 meta["vex"] = file 
            elif ( ( vex_file[0:1] == "/"  and path + "/"  + file == v2d_file ) or \
                     file == v2d_file ):
#
# -------------- Get the name of the v2d file
#
                 meta["v2d"] = file 
            elif ( len(file) > 8 ):
#
# -------------- Put names of DiFX input file in one list
#
                 if ( file[len(file)-6:len(file)] == ".input" ):
                      input_file_list.append ( path + "/" + file )
                 else:
#
# ------------------- ... and in file_list all other names
#
                      file_list.append ( path + "/" + file )


    tar_exc = ""
#
# --- Check whether vex and v2d files have been found
#
    if ( not "vex" in meta.keys() ):
         print ( "Did not find vex file %s in directory %s" % ( vex_file, dirin ) )
         if ( verb > 2 ): 
              for line in meta.keys(): 
                  print ( "meta: ", line )
         return ( 1, None )
    if ( not "v2d" in meta.keys() ):
         print ( "Did not find v2d file %s in directory %s" % ( v2d_file, dirin ) )
         return ( 1, None )

#
# -- Sort file lists
#
    file_list.sort()
    input_file_list.sort()
#
# --- Check for widowed difx input file
#
    check_err_list = []
    for inp_file in input_file_list:
        difx_dir  = inp_file.replace(".input",".difx")
        calc_file = inp_file.replace(".input",".calc")
        im_file   = inp_file.replace(".input",".im")
        flag_file = inp_file.replace(".input",".flag")
#
        difx_dir_status  = "??"
        calc_file_status = "??" 
        im_file_status   = "??" 
        flag_file_status = "??" 
#
# ----- We need find matching files to a given input DiFX file
#
        for file in file_list:
            if ( difx_dir + "/DIFX" in file and ".b00" in file ):
                 difx_dir_status = "OK"
            elif ( file == calc_file ):
                 calc_file_status = "OK"
            elif ( file == im_file ):
                 im_file_status = "OK"
                 im_file_list.append ( file )
            elif ( file == flag_file ):
                 flag_file_status = "OK"

        if ( difx_dir_status != "OK" ):
             check_err_list.append ( "Difx input file %s does not have a corresponding " \
                      "binary visibility file in the input directory %s" % \
                      ( inp_file, dirin ) )
        if ( calc_file_status != "OK" ):
             check_err_list.append ( "Difx input file %s does not have a corresponding calc file " \
                      "in the input directory %s" % ( inp_file, dirin) )
        if ( im_file_status != "OK" ):
             check_err_list.append ( "Difx input file %s does not have a corresponding im file " \
                      "in the input directory %s" % ( inp_file, dirin ) )
        if ( flag_file_status != "OK" ):
             check_err_list.append ( "Difx input file %s does not have a corresponding flag file " \
                      "in the input directory %s" % ( inp_file, dirin ) )

    if ( len(check_err_list) > 0 ):
#
# ------ Generate an error file
#
         err_file = tmp_dir + "/" + meta["exper_name"] + "_err.txt"
         f = open ( err_file, "w" )
         for rec in check_err_list:
             print ( rec, file=f )
         f.close()
         print ( "%d missing files were found. Please, see their list in %s" % \
                 ( len(check_err_list), err_file ) ) 
         print ( "You need either recover missing files, or rename " \
                 "affected input files to extensions .missing_input" )
         return ( 1, None )
    else:
#
# ------ Remove a stale error file (if we can)
#
         err_file     = tmp_dir + "/" + meta["exper_name"] + "_err.txt"
         if ( os.path.isfile ( err_file ) ):
              try:  
                    os.remove ( meta_file )
              except:
                    pass

    tim_min =  1.e15
    tim_max = -1.e15
    mjd_min = 0
    mjd_max = 0
    utc_min = 0
    utc_max = 0
    if ( verb > 1 ):
         print ( "Parsing input DiFX files" )
#
# --- Parse all DiFX input files
#
    for input_file in input_file_list:
        inp = read_file ( input_file )
#
# ----- Extract a number of keywords from the DiFX input file
#
        for line in inp:
            if ( line[0:10] == "START MJD:" ):
                 mjd = int(line.split()[2])
            if ( line[0:14] == "START SECONDS:" ):
                 utc_sec = int(line.split()[2])
                 if ( mjd*86400 + utc_sec < tim_min ):
                      mjd_min = mjd
                      utc_min = utc_sec
                      tim_min = mjd*86400 + utc_sec 
                 if ( mjd*86400 + utc_sec > tim_max ):
                      mjd_max = mjd
                      utc_max = utc_sec
                      tim_max = mjd*86400 + utc_sec 
            if ( line[0:14] == "TELESCOPE NAME" ):
                 tel = line.split()[3]
                 if ( not tel in tel_list ):
                      tel_list.append(tel)
            if ( "REC BAND" in line and "POL:" in line ):
                 pol = line.split()[4]
                 if ( not pol in pol_list ):
                      pol_list.append(pol)

    im_file_list.sort()
    for im_file in im_file_list:
        im = read_file ( im_file )
        for line in im:
            line = line.replace(":"," ")
            if ( "SCAN" in line and "POINTING" in line and len(line.split()) == 5 ):
                 sou = line.split()[4] 
                 if ( not sou in sou_list ):
                      sou_list.append ( sou )
            if ( "SCAN" in line and "PHS STR"  in line and len(line.split()) == 6 ):
                 sou = line.split()[5] 
                 if ( not sou in sou_list ):
                      sou_list.append ( sou )

    sou_list.sort()

#
# --- ... and put them in meta dictionary
#
    meta["num_input"] = len(input_file_list)
    meta["tels"]      = tel_list
    meta["pols"]      = pol_list
    meta["sous"]      = sou_list
    meta["mjd_min"]   = mjd_min
    meta["mjd_max"]   = mjd_max
    meta["utc_min"]   = utc_min
    meta["utc_max"]   = utc_max

#
# --- Write down collected information in the output metadata file
#
    try:
        f=open ( meta_file, "w" )
    except BaseException as e: 
        print ( "Error in attemptting to write file %s -- %s" % ( meta_name, str(e) ) )

    print ( "# SWIN-ARCHIVE meta data.  Format version  1.01  2020.11.11", file=f )
    print ( "# ", file=f )
    print ( "# Generated on %s UTC" % datetime.datetime.utcnow().strftime ( "%Y.%m.%d_%H:%M:%S" ), file=f  )
    print ( "# Generated by %s" % vsdc_swin_gentar__label, file=f )
    print ( "# Generated with %s" % vsdc.invoked_as, file=f )
    print ( "# From the current directory %s" % vsdc.current_dir, file=f )
    print ( "# ", file=f )
    print ( "exper_name: %s" % meta["exper_name"], file=f )
    print ( "corr_vers:  %d" % corr_vers, file=f )
    print ( "exper_desc: %s" % meta["exper_desc"], file=f )
    print ( "pi_name:    %s" % meta["pi_name"], file=f    )
    print ( "corr_name:  %s" % meta["target_correlator"], file=f )
    print ( "date_start: %5d  %5d  %s UTC" % ( mjd_min, utc_min, \
                         mjd_tim_to_time ( mjd_min, utc_min).strftime ( "%Y.%m.%d_%H:%M:%S" ) ), file=f )
    print ( "date_stop:  %5d  %5d  %s UTC" % ( mjd_max, utc_max, \
                         mjd_tim_to_time ( mjd_max, utc_max).strftime ( "%Y.%m.%d_%H:%M:%S" ) ), file=f )
    print ( "duration:   %5d  sec" % ( (mjd_max - mjd_min)*86400 + (utc_max - utc_min) ), file=f )
    print ( "num_sta:     %-5d " % len(meta["tels"]), file=f )
    print ( "num_sou:    %-5d "  % len(meta["sous"]), file=f )
    print ( "stations:   %s " % " ".join(meta["tels"]), file=f )
    print ( "sources:    %s " % " ".join(meta["sous"]), file=f )
    print ( "polariz:    %s " % " ".join(meta["pols"]), file=f )
    print ( "num_inps:   %d " % len(input_file_list), file=f )
    print ( "num_files:  %d " % ( len(file_list) + 1), file=f )
    print ( "vex_file:   %s " % vex_file.replace(dirin+"/",""), file=f )
    print ( "v2d_file:   %s " % v2d_file.replace(dirin+"/",""), file=f )
    print ( "#", file=f )
    print ( "file:      ", meta_file.replace(tmp_dir+"/",""), file=f )
    print ( "file:      ", vex_file.replace(dirin+"/",""), file=f  )
    print ( "file:      ", v2d_file.replace(dirin+"/",""), file=f  )

    filout = dirout + "/" + \
             mjd_tim_to_time ( mjd_min, utc_min).strftime("%Y%m%d")  + "_" + \
             meta["exper_name"] + "_" + \
             "v%03d_swin.tar" % corr_vers 
#
# --- Generate a command for data compression
#

    if ( len(vsdc.tar_swin_exclude) > 0 ):
#
# ------ Add tar excliude options
#
         tar_exc = ""
         for line in vsdc.tar_swin_exclude:
             tar_exc = tar_exc + " --exclude " + line
    
    if ( compr == "none" ):
         output_file = filout
         tar_com = "umask 2; tar cvf " + output_file + " " + \
                   tar_exc + " " + \
                   "-C " + tmp_dir + " " + meta["exper_name"] + "_meta.txt" + " " + \
                   "-C " + dirin + "/" + " -T " + tar_list_fil
    elif ( compr == "bzip2" ):
         output_file = filout + ".bz2"
         tar_com = "tar cvf " + output_file + " -I bzip2" + " " + \
                   tar_exc + " " + \
                   "-C " + tmp_dir + " " + meta["exper_name"] + "_meta.txt" + " " + \
                   "-C " + dirin  + "/" + " -T " + tar_list_file
         compr_check = ( "bzip2 --help" )
         (ret,out) = vsdc_exe ( compr_check )
         if ( ret != 0 ):
              print  ( "vsdc_swin: compressor program bzip2 does not run: ", out[0] )        
              return ( 1, None )
    elif ( compr == "lbzip2" ):
         output_file = filout + ".bz2"
         tar_com = "tar cvf " + output_file + " -I lbzip2" + " " + \
                   tar_exc + " " + \
                   "-C " + tmp_dir + " " + meta["exper_name"] + "_meta.txt" + " " + \
                   "-C " + dirin + "/" + " -T " + tar_list_file
         compr_check = ( "lbzip2 --help" )
         (ret,out) = vsdc_exe ( compr_check )
         if ( ret != 0 ):
              print  ( "vsdc_swin: compressor program lbzip2 does not run: ", out[0] )        
              return ( 1, None )
    elif ( compr == "gzip" ):
         output_file = filout + ".gz"
         tar_com = "tar cvf " + output_file + " -I gzip" + " " + \
                   tar_exc + " " + \
                   "-C " + tmp_dir + " " + meta["exper_name"] + "_meta.txt" + " " + \
                   "-C " + dirin + "/" + " -T " + tar_list_file
         compr_check = ( "gzip --help" )
         (ret,out) = vsdc_exe ( compr_check )
         if ( ret != 0 ):
              print  ( "vsdc_swin: compressor program gzip does not run: ", out[0] )        
              return ( 1, None )
    else:
         print ( "vsdc_swin: unsupported compr variable. Expected: none bzip2 lbzip2 gzip, but got ", compr )
         return ( 1, None )

    if ( verb == 0 ): 
#
# ------ Remove verbosity from tar
#
         tar_com = tar_com.replace("tar cvf ","tar cf ")

#
# --- At this point the file list is empty.
# --- Append vex_file, v2d_file to the file list of the files to be archived. 
#

    tar_list = []
    tar_list.append ( vex_file.replace(dirin+"/","")    )
    tar_list.append ( v2d_file.replace(dirin+"/","")    )

#
# --- Append input files to the the file list of the files to be archived. 
#
    for line in input_file_list:
        tar_list.append ( line.replace(dirin+"/","") )
        print ( "file:      ", line.replace(dirin+"/",""), file=f )

#
# --- Append all remaining files to the the file list of the files to be archived. 
#
    for line in file_list:
        tar_list.append ( line.replace(dirin+"/","") )
        print ( "file:      ", line.replace(dirin+"/",""), file=f )
    f.close()
#
# --- Write the tar list file
#
    f = open(tar_list_file,"w")
    for line in tar_list:
        print ( line, file=f )
    f.close()
#
# --- Run the tar command
#
    if ( verb > 1 ):
         print ( "Executing comand ", tar_com )
    (ret,out) = vsdc_exe_pipe ( tar_com )
    if ( verb < 4 ):
         try:
              os.remove ( meta_file )
              os.remove ( tar_list_file )
         except:
              pass
    if ( ret !=0 ):
         for line in out:
             print ( line )
         print ( "vsdc_swin_gentar: failed to generate the tarred archive file. ret= ", ret )
         print ( "vsdc_swin_gentar: len(tar_com) = ", len(out) ) 
         print ( tar_com )
         return ( 1, None )
    if ( verb >=2 ):
         print ( tar_com )
    if ( verb >=1 ):
         print ( "Output file created: " + output_file )
    return ( 0, output_file )

#
# ------------------------------------------------------------------------
#
def vsdc_swin_check_server ( swin_magic, swin_tar_file ):
    """
    vsdc_swin_check_server returns a tuple of five variables on exit:

    -- Flag success (True) or failure (False)
    -- message
    -- experiment name;
    -- experiment start date in Python format;
    -- experiment stop  date in Python format;

    If the file passed the check, the message is "OK". Any other message means
    the check failed and fields experiment name, experiment start, experiment 
    stop may be None.
    """
    try:
          (ret, exper_name, date_start, date_stop ) = vsdc_swin_check ( swin_magic, swin_tar_file )
    except Exception as e: 
           ret = "Unknown Error Occurred in vsdc_swin_check: " + str(e)
    if ( ret == "OK" ):
         return [True, None, exper_name, date_start, date_stop]
    else: 
         return [False, "FATAL: Error Occurred in vsdc_swin_check: " + ret, \
                 None, None, None ]

