      PROGRAM    ERM_TO_EOP_MAIN
! ************************************************************************
! *                                                                      *
! *   Program ERM_TO_EOP computes the EOP series based on the            *
! *   coefficients of the empirical Earth Rotation Model and the         *
! *   harmonic. EOP expansion that are estimated in a global VLBI        *
! *   solution. The output file may have an arbitrary step and cover     *
! *   the specified time range. However, that range cannot go beyond     *
! *   the range of the empirical Earth Rotation Model.                   *
! *                                                                      *
! *   Usage:                                                             *
! *                                                                      *
! *   erm_to_eop erm_file heo_file beg_date end_date step eob_file       *
! *                                                                      *
! *   erm_file -- file with the output Empirical Earth Rotation Model    *
! *               generated by getpar from a listing of pSolve VLBI      *
! *               solution.                                              *
! *                                                                      *
! *   heo_file -- file with the coefficients of the harmonic Earth       *
! *               orientation parameters generated by getpar from        *
! *               a listing of pSolve VLBI solution. Or no               *
! *                                                                      *
! *   beg_date -- start date of the output time series. It can be in     *
! *               YYYY.DD.MM_hh:mm:ss.s format of as a string beg.       *
! *               If string beg is specfied, the start date of the       *
! *               output series is set to the start date of the          *
! *               empirical Earth rotation model.                        *
! *                                                                      *
! *   end_date -- end date of the output time series. It can be in       *
! *               YYYY.DD.MM_hh:mm:ss.s format of as a string end.       *
! *               If string end is specfied, the end date of the         *
! *               output series is determined.                           *
! *                                                                      *
! *   step     -- Time step in seconds of the output time series.        *
! *                                                                      *
! *   eob_file -- Output file in EOB format.                             *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ### 13-OCT-2021 ERM_TO_EOP_MAIN v1.1 (d) L. Petrov 30-OCT-2021 ###  *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      INCLUDE   'astro_constants.i'
      INCLUDE   'vtd.i'
      INCLUDE   'getpar.i'
      INCLUDE   'erm.i'
      TYPE      ( ERM__TYPE  ) :: ERM
      TYPE      ( EOB__CHAR ) ::  EOB
      CHARACTER  FIL_ERM*128, FIL_HEO*128, FIL_EOB*128, STR*128, STR1*128, &
     &           BEG_DAT*32, END_DAT*32, STEP_STR*32, GET_VERSION*54
      LOGICAL*1  LEX
      REAL*8     TIM_BEG, TIM_END, TIM_STP, TIM_EPS, SEC
      INTEGER*4  M_ERM, M_HEO, MO
      PARAMETER  ( M_ERM = 32768 )
      PARAMETER  ( M_HEO =  2048 )
      PARAMETER  ( MO    = 1024*1024 )
      TYPE ( EOP__STRU  ) :: EOP(M_ERM)
      TYPE ( HEO__STRUC ) :: HEO(M_HEO)
      REAL*8     HEO_EPOCH_SEC, TARG_TDB, UT1_M_TDB, &
     &           E1_MHB, E2_MHB, E1_WAHR, E2_WAHR, &
     &           DPSI_MHB, DEPS_MHB, DPSI_WAHR, DEPS_WAHR, DPSI, DEPS, &
     &           E1_RATE, E2_RATE, DPSI_RATE, DEPS_RATE, &
     &           CROSS_NUT_E3, TAI, TDB
      PARAMETER  ( TIM_EPS = 1.0D-6 )
      CHARACTER  NAME_HEO(M_HEO)*64, OUT(MO)*512
      INTEGER*4  J1, N_ERM(3), N_EOP, L_HEO, UZT_MODEL, UZT_USE, &
     &           MJD, IVRB, NO, IUER
      CHARACTER, EXTERNAL :: GET_CDATE*19, TIM_TO_DATE*23
!
      INCLUDE   'erm_to_eop_version.i'
      UZT_MODEL = UZT__NONE
      UZT_USE   = UZT__NONE
!
      IVRB = 0
      IF ( IARGC() < 6 ) THEN
           WRITE ( 6, '(A)' ) 'Usage: erm_to_eop erm_file heo_file beg_date end_date step eob_file'
           CALL EXIT ( 1 )
         ELSE
           CALL GETARG ( 1, FIL_ERM  )
           CALL GETARG ( 2, FIL_HEO  )
           CALL GETARG ( 3, BEG_DAT  )
           CALL GETARG ( 4, END_DAT  )
           CALL GETARG ( 5, STEP_STR )
           CALL GETARG ( 6, FIL_EOB  )
      END IF
!
      INQUIRE ( FILE=FIL_ERM, EXIST=LEX )
      IF ( .NOT. LEX ) THEN
           IUER = -1
           CALL ERR_LOG ( 7601, IUER, 'ERM_TO_EOP_MAIN', 'Cannot find '// &
     &         'input ERM file '//FIL_ERM )
           CALL EXIT ( 1 )
      END IF
!
      IF ( .NOT. ( FIL_HEO == 'no' .OR. FIL_HEO == 'NO' ) ) THEN
           INQUIRE ( FILE=FIL_HEO, EXIST=LEX )
           IF ( .NOT. LEX ) THEN
                IUER = -1
                CALL ERR_LOG ( 7602, IUER, 'ERM_TO_EOP_MAIN', 'Cannot find '// &
          &         'input HEO file '//FIL_HEO )
                CALL EXIT ( 1 )
           END IF
         ELSE
           CONTINUE 
      END IF
!
      IF ( BEG_DAT(1:3) == 'beg' ) THEN
           TIM_BEG = 1.01D10
         ELSE
           IUER = -1
           CALL DATE_TO_TIME ( BEG_DAT, MJD, SEC, IUER )
           IF ( IUER .NE. 0 ) THEN
                IUER = -1
                CALL ERR_LOG ( 7603, IUER, 'ERM_TO_EOP_MAIN', 'Error in parsing '// &
     &              'beg_date '//BEG_DAT )
                CALL EXIT ( 1 )
           END IF
           TIM_BEG = (MJD - J2000__MJD)*86400.0D0 + SEC
      END IF
!

      IF ( END_DAT(1:3) == 'end' ) THEN
           TIM_END = -1.01D10
         ELSE
           IUER = -1
           CALL DATE_TO_TIME ( END_DAT, MJD, SEC, IUER )
           IF ( IUER .NE. 0 ) THEN
                IUER = -1
                CALL ERR_LOG ( 7604, IUER, 'ERM_TO_EOP_MAIN', 'Error in parsing '// &
     &              'end_date '//END_DAT )
                CALL EXIT ( 1 )
           END IF
           TIM_END = (MJD - J2000__MJD)*86400.0D0 + SEC
      END IF
!
      IF ( INDEX ( STEP_STR, '.' ) < 1 ) THEN
           STEP_STR = TRIM(STEP_STR)//'.0'
      END IF 
!
      READ ( UNIT=STEP_STR, FMT='(F10.1)', IOSTAT=IUER ) TIM_STP
      IF ( IUER .NE. 0 ) THEN
           IUER = -1
           CALL ERR_LOG ( 7605, IUER, 'ERM_TO_EOP_MAIN', 'Error in '// &
     &         'parsing time step argument '//TRIM(STEP_STR)// &
     &         ' -- a real number was expected' )
           CALL EXIT ( 1 )
      END IF
!
      IUER = -1
      CALL PARSE_ERM ( FIL_ERM, ERM, IUER )
      IF ( IUER .NE. 0 ) THEN
           IUER = -1
           CALL ERR_LOG ( 7606, IUER, 'ERM_TO_EOP_MAIN', 'Error in '// &
     &         'reading input empirical Earth rotation model from file '// &
     &          FIL_ERM )
           RETURN 
      END IF
!
      IF ( .NOT. ( FIL_HEO == 'no' .OR. FIL_HEO == 'NO' ) ) THEN
           IUER = -1
           CALL READ_HEO ( FIL_HEO, M_HEO, L_HEO, HEO, NAME_HEO, HEO_EPOCH_SEC, &
     &                     IUER )
           IF ( IUER .NE. 0 ) THEN
                IUER = -1
                CALL ERR_LOG ( 7607, IUER, 'ERM_TO_EOP_MAIN', 'Error in '// &
     &              'reading input file with estimates of harmonic Earth  '// &
     &              'rotation model '//FIL_HEO )
                RETURN 
           END IF
         ELSE 
           L_HEO = 0
           HEO_EPOCH_SEC = 0.0D0
      END IF
!
      IF ( TIM_BEG < (ERM%MJD_BEG - J2000__MJD)*86400.0D0 + ERM%TAI_BEG ) THEN
           TIM_BEG = (ERM%MJD_BEG - J2000__MJD)*86400.0D0 + ERM%TAI_BEG
      END IF
!
      IF ( TIM_END > (ERM%MJD_END - J2000__MJD)*86400.0D0 + ERM%TAI_END - ERM%TIME_EST_SPAN(3) ) THEN
           TIM_END = (ERM%MJD_END - J2000__MJD)*86400.0D0 + ERM%TAI_END - ERM%TIME_EST_SPAN(3)
      END IF
!
      IUER = -1
      CALL ERM_TO_EOP ( ERM, TIM_STP, UZT_MODEL, UZT_USE, &
     &                  M_ERM, N_EOP, EOP, L_HEO, HEO, &
     &                  HEO_EPOCH_SEC, IVRB, IUER )
      IF ( IUER .NE. 0 ) THEN
           IUER = -1
           CALL ERR_LOG ( 7608, IUER, 'ERM_TO_EOP_MAIN', 'Error in '// &
     &         'computation of the EOP files using coefficients of '// &
     &         'the empirical Earth rotation model' )
           CALL EXIT ( 1 )
      END IF
!
! --- Writing EOP in EOB-format
!
      IUER = -1
      STR  = TIM_TO_DATE ( TIM_BEG, IUER )
      IUER = -1
      STR1 = TIM_TO_DATE ( TIM_END, IUER )
      NO = 0
      NO = NO + 1 ; OUT(NO) = SIG_EOB
      NO = NO + 1 ; OUT(NO) = '# Earth orientation parameters from the VLBI solution '//TRIM(ERM%SOL_ID)
      NO = NO + 1 ; OUT(NO) = '# Generated by '//TRIM(GET_VERSION())//' on '//GET_CDATE()
      NO = NO + 1 ; OUT(NO) = '# For the time interval [ '//STR(1:19)//', '//STR1(1:19)//' ]'
      NO = NO + 1 ; OUT(NO) = '#'
      NO = NO + 1 ; OUT(NO) = '# EOP totals DO NOT include the effect of subdaily variations'
      NO = NO + 1 ; OUT(NO) = '# Time argument: TAI'
      NO = NO + 1 ; OUT(NO) = '# Nutation angles are wrt Wahr1980 expansion'
      NO = NO + 1 ; OUT(NO) = '#'

      DO 410 J1=1,N_EOP
         IF ( (EOP(J1)%MJD_EOP - J2000__MJD)*86400.0D0 + TIM_EPS < TIM_BEG ) GOTO 410
         IF ( (EOP(J1)%MJD_EOP - J2000__MJD)*86400.0D0 - TIM_EPS > TIM_END ) GOTO 410
         NO = NO + 1
         CALL CLRCH  ( OUT(NO) )
!
         CALL LIB$MOVC3 ( SIZEOF(EOB), %REF(OUT(NO)), EOB )
         EOB%SCODE  = ' '
         EOB%FLAG   = ' '
         WRITE ( UNIT=EOB%DBNAME, FMT='("g_",F8.2)' ) EOP(J1)%MJD_EOP
!
         CALL TAI_TO_TDB  ( IDINT(EOP(J1)%MJD_EOP), 0.0D0, TDB )
         TARG_TDB  = (EOP(J1)%MJD_EOP - J2000__MJD - 0.5D0)*86400.0D0 + TDB
         UT1_M_TDB = 0.0D0
         CALL HEO_MHB2000  ( 4, TARG_TDB, UT1_M_TDB, E1_MHB, E2_MHB, &
     &                       DPSI_MHB, DEPS_MHB, &
     &                       E1_RATE, E2_RATE, DPSI_RATE, DEPS_RATE, &
     &                       CROSS_NUT_E3  )
         CALL HEO_WAHR1980 ( 4, TARG_TDB, UT1_M_TDB, E1_WAHR, E2_WAHR, &
     &                       DPSI_WAHR, DEPS_WAHR, &
     &                       E1_RATE, E2_RATE, DPSI_RATE, DEPS_RATE, &
     &                       CROSS_NUT_E3  )
!
         WRITE ( UNIT=EOB%MJD_EOP, FMT='(F12.6)' ) EOP(J1)%MJD_EOP
         WRITE ( UNIT=EOB%MJD_NUT, FMT='(F12.6)' ) EOP(J1)%MJD_NUT
         WRITE ( UNIT=EOB%XPL_V,   FMT='(F8.6)'  ) EOP(J1)%XPL_V*RAD__TO__ARCSEC
         WRITE ( UNIT=EOB%YPL_V,   FMT='(F8.6)'  ) EOP(J1)%YPL_V*RAD__TO__ARCSEC
         WRITE ( UNIT=EOB%U1_V,    FMT='(F11.7)' ) EOP(J1)%U1_V*RAD__TO__SEC
         WRITE ( UNIT=EOB%DPSI_V,  FMT='(F8.3)'  ) (EOP(J1)%DPSI_V + DPSI_MHB - DPSI_WAHR)*RAD__TO__MAS
         WRITE ( UNIT=EOB%DEPS_V,  FMT='(F8.3)'  ) (EOP(J1)%DEPS_V + DEPS_MHB - DEPS_WAHR)*RAD__TO__MAS
         WRITE ( UNIT=EOB%XPR_V,   FMT='(F9.6)'  ) EOP(J1)%XPR_V*RAD__TO__ARCSEC*86400.0D0
         WRITE ( UNIT=EOB%YPR_V,   FMT='(F9.6)'  ) EOP(J1)%YPR_V*RAD__TO__ARCSEC*86400.0D0
         WRITE ( UNIT=EOB%UTR_V,   FMT='(F7.4)'  ) EOP(J1)%UTR_V*RAD__TO__SEC*86400.0D0*1.D3
!
         WRITE ( UNIT=EOB%XPL_E,   FMT='(F8.6)'  ) EOP(J1)%XPL_E*RAD__TO__ARCSEC
         WRITE ( UNIT=EOB%YPL_E,   FMT='(F8.6)'  ) EOP(J1)%YPL_E*RAD__TO__ARCSEC
         WRITE ( UNIT=EOB%U1_E,    FMT='(F9.7)'  ) EOP(J1)%U1_E*RAD__TO__SEC
         WRITE ( UNIT=EOB%DPSI_E,  FMT='(F7.3)'  ) EOP(J1)%DPSI_E*RAD__TO__MAS
         WRITE ( UNIT=EOB%DEPS_E,  FMT='(F7.3)'  ) EOP(J1)%DEPS_E*RAD__TO__MAS
         WRITE ( UNIT=EOB%XPR_E,   FMT='(F9.6)'  ) 0.0*RAD__TO__ARCSEC*86400.0D0
         WRITE ( UNIT=EOB%YPR_E,   FMT='(F9.6)'  ) 0.0*RAD__TO__ARCSEC*86400.0D0
         WRITE ( UNIT=EOB%UTR_E,   FMT='(F7.4)'  ) 0.0*RAD__TO__SEC*86400.0D0*1.D3
!
         CALL LIB$MOVC3 ( SIZEOF(EOB), EOB, %REF(OUT(NO)) )
 410  CONTINUE 
!
      IUER = -1
      CALL WR_TEXT ( NO, OUT, FIL_EOB, IUER )
      IF ( IUER .NE. 0 ) THEN
           IUER = -1
           CALL ERR_LOG ( 7609, IUER, 'ERM_TO_EOP_MAIN', 'Error in '// &
     &         'attempt to write the EOP in the output file '//FIL_EOB )
           CALL EXIT ( 1 )
      END IF
!
      END   PROGRAM  ERM_TO_EOP_MAIN  !#!#
