      SUBROUTINE PIMA_TSMO ( PIM, MODE, IUER )
! ************************************************************************
! *                                                                      *
! *   Routine PIMA_TSMO computes model of the system temperature. It     *
! *   supported two modes: "if" and "elev". See comments of subroutines  *
! *   PIMA_TSMO_IF and PIMA_TSMO_ELEV for details.                       *
! *                                                                      *
! *   Important: opacity data should be generated by PIMA task opag      *
! *   and loaded by task opal before using PIMA_TSMO.                    *
! *   
! * ________________________ Input parameters: _________________________ *
! *                                                                      *
! * MODE ( CHARACTER ) -- String of comma-separated modes. Supported     *
! *                       modes: if, elev.                               *
! *                                                                      *
! * ________________________ Modified parameters: ______________________ *
! *                                                                      *
! *       PIM ( PIMA__TYP ) -- Object with information related to        *
! *                            program PIMA.                             *
! * IUER ( INTEGER*4, OPT ) -- Universal error handler.                  *
! *                      Input: IUER=0 -- no error message will be       *
! *                                       printed even in the case       *
! *                                       of error.                      *
! *                             IUER=-1,-2,-3 -- in the case of error    *
! *                                       the message will be put on     *
! *                                       stdout.                        *
! *                             IUER=-3 -- in the case of error after    *
! *                                        printing the error message    *
! *                                        the program will terminate.   *
! *                       Output:                                        *
! *                             if input value of IUER =-2,-3 -- IUER    *
! *                                        is not modified.              *
! *                             otherwise, the output value of IUER is 0 *
! *                             in the case of successful and            *
! *                             positive non-zero in the vase of errors. *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ### 17-AUG-2017   PIMA_TSMO   v1.0 (d)  L. Petrov  31-AUG-2017 ###  *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      INCLUDE   'astro_constants.i'
      INCLUDE   'pima.i'
      TYPE     ( PIMA__TYPE  ) :: PIM
      CHARACTER  MODE*(*)
      INTEGER*4  IUER
      INTEGER*4  MIND
      PARAMETER  ( MIND = 16 )
      INTEGER*4  J1, J2, J3, J4, J5, J6, IND(2,MIND), LIND, K_STA, KPOL, IER
      LOGICAL*1  FL_IF, FL_ELEV
      INTEGER*4, EXTERNAL :: ILEN, I_LEN
!
! --- Parse the mode string
!
      IER = -1
      CALL EXWORD ( MODE, MIND, LIND, IND, ', '//CHAR(0), IER )
      FL_IF   = .FALSE.
      FL_ELEV = .FALSE.
      IF ( LIND .GE. 1 ) THEN
           IF ( MODE(IND(1,1):IND(2,1)) == 'if'   ) FL_IF   = .TRUE.
           IF ( MODE(IND(1,1):IND(2,1)) == 'elev' ) FL_ELEV = .TRUE.
      END IF
      IF ( LIND .GE. 2 ) THEN
           IF ( MODE(IND(1,2):IND(2,2)) == 'if'   ) FL_IF   = .TRUE.
           IF ( MODE(IND(1,2):IND(2,2)) == 'elev' ) FL_ELEV = .TRUE.
      END IF
      IF ( PIM%NPOL == 1 ) THEN
           KPOL = 1
         ELSE
           KPOL = 2
      END IF
!
      IF ( FL_IF ) THEN
!
! -------- IF model 
!
           DO 410 J1=1,KPOL
              CALL ERR_PASS ( IUER, IER )
              CALL PIMA_TSMO_IF ( PIM, J1, K_STA, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_PASS ( IUER, IER )
                   CALL ERR_LOG ( 4111, IER, 'PIMA_TSMO', 'Error in attempt '// &
     &                 'to model Tsys noise between IFs.  Netherless, continue.' )
                   RETURN 
              END IF
              IF ( K_STA == 0 ) THEN
                   CALL ERR_LOG ( 4112, IUER, 'PIMA_TSMO', 'Tsys and opacity was '// &
     &                 'not loaded for any station. Please load tsys and run tasks '// &
     &                 'opag and opal' )
                   RETURN 
              END IF
              IF ( PIM%CONF%DEBUG_LEVEL .GE. 1 ) THEN
                   WRITE ( 6, 110 ) PIMA__POL(J1), K_STA
 110               FORMAT ( 'PIMA_TSMO "if"   mode, ', A, ' polarization, ', &
     &                       I2, ' stations were processed' )
              END IF
 410       CONTINUE 
      END IF
      IF ( FL_ELEV ) THEN
!
! -------- elevation mode
!
           DO 420 J2=1,KPOL
              CALL ERR_PASS ( IUER, IER )
              CALL PIMA_TSMO_ELEV ( PIM, K_STA, J2, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_PASS ( IUER, IER )
                   CALL ERR_LOG ( 4113, IER, 'PIMA_TSMO', 'Error in attempt '// &
     &                 'to model elevation dependence of Tsys.  Netherless, continue.' )
                   RETURN 
              END IF
              IF ( K_STA == 0 ) THEN
                   CALL ERR_LOG ( 4114, IUER, 'PIMA_TSMO', 'Tsys and opacity was '// &
     &                 'not loaded for any station. Please load tsys and run tasks '// &
     &                 'opag and opal' )
                   RETURN 
              END IF
              IF ( PIM%CONF%DEBUG_LEVEL .GE. 1 ) THEN
                   WRITE ( 6, 120 ) PIMA__POL(J2), K_STA
 120               FORMAT ( 'PIMA_TSMO "elev" mode, ', A, ' polarization, ', &
     &                       I2, ' stations were processed' )
              END IF
 420       CONTINUE 
      END IF
!      
      CALL ERR_LOG ( 0, IUER )
      RETURN
      END  SUBROUTINE  PIMA_TSMO  !#!  
!
! ------------------------------------------------------------------------
!
      SUBROUTINE PIMA_TSMO_IF ( PIM, IND_POL, K_STA, IUER )
! ************************************************************************
! *                                                                      *
! *   Routine PIMA_TSMO_IF performs analysis of Tsys ratios between      *
! *   all IFs of the specified band and filters out outliers. It writes  *
! *   the corrected Tsys in PIM%STA()%STMO  object.                      *
! *   PIMA_TSMO in the "if" mode computes the mean ratio of Tsys between *
! *   IFs of the band specified by FRQ_GRP, BEG_FRQ and END_FRQ          *
! *   keywords. PIMA_TSMO assumes the ratios should be stable within     *
! *   an experiment. It finds the reference IF that has less outliers.   *
! *   It iteratively removes outliers that exceed rms*FRIB.NOISE_NSIGMA. *
! *   Results are written in STMO objects in arrays TSYS_MOD and         *
! *   TSYS_CLN.                                                          *
! *                                                                      *
! *   For the reference IF, TSYS_MOD is the measured Tsys unless it is   *
! *   an outlier. If an outlier happened at the reference IF,            *
! *   a temporarily reference IF for that observations is found.         *
! *   For non-referenced IFs, TSYS_MOD is the Tsys in the reference IF   *
! *   multiplied by the mean ratio of the Tsys at the remote IF to the   *
! *   reference IF.                                                      *
! *                                                                      *
! *   If a given measured is not an outlier, TSYS_CLN is measured        *
! *   Tsys. Otherwise, TSYS_CLN is set to be equal to TSYS_MOD for       *
! *   a given IF, given observation.                                     *
! *                                                                      *
! * ________________________ Input parameters: _________________________ *
! *                                                                      *
! * IND_POL ( INTEGER*4 ) -- Polarization index: 1 for R and 2 for L.    *
! *                                                                      *
! * ________________________ Output parameters: ________________________ *
! *                                                                      *
! *                                                                      *
! *   K_STA ( INTEGER*4 ) -- The number of processed station that have   *
! *                          input Tsys.                                 *
! *                                                                      *
! * ________________________ Modified parameters: ______________________ *
! *                                                                      *
! *       PIM ( PIMA__TYP ) -- Object with information related to        *
! *                            program PIMA.                             *
! * IUER ( INTEGER*4, OPT ) -- Universal error handler.                  *
! *                      Input: IUER=0 -- no error message will be       *
! *                                       printed even in the case       *
! *                                       of error.                      *
! *                             IUER=-1,-2,-3 -- in the case of error    *
! *                                       the message will be put on     *
! *                                       stdout.                        *
! *                             IUER=-3 -- in the case of error after    *
! *                                        printing the error message    *
! *                                        the program will terminate.   *
! *                       Output:                                        *
! *                             if input value of IUER =-2,-3 -- IUER    *
! *                                        is not modified.              *
! *                             otherwise, the output value of IUER is 0 *
! *                             in the case of successful and            *
! *                             positive non-zero in the vase of errors. *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ### 17-AUG-2017  PIMA_TSMO_IF  v1.3 (d)  L. Petrov  24-FEB-2025 ### *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      INCLUDE   'astro_constants.i'
      INCLUDE   'pima.i'
      TYPE     ( PIMA__TYPE  ) :: PIM
      REAL*8       PIMA__SCM, EL_LIM_MIN, TSYS_LIM_MIN, TSYS_LIM_MAX 
      PARAMETER  ( PIMA__SCM = 1.25D0 )
      PARAMETER  ( EL_LIM_MIN   = 2.0D0*DEG__TO__RAD )
      PARAMETER  ( TSYS_LIM_MIN = 10.0D0 )
      PARAMETER  ( TSYS_LIM_MAX = 950.0D0 )
      REAL*8,    ALLOCATABLE :: TSVAL_IN(:,:), TSVAL_OUT(:,:)
      LOGICAL*1, ALLOCATABLE :: TSFLG(:,:)
      CHARACTER  POLAR_STR*2, STR*128
      REAL*8     TSYS_VAL, TIM_DIF_MIN, TSRAT(PIM__MFRQ,PIM__MSTA), &
     &           TSRMS(PIM__MFRQ,PIM__MSTA), TIM_STMO
      LOGICAL*1  FL_TSYS_TOUSE, FL_TSYS_OK
      INTEGER*4  K_STA, IND_POL, IUER
      INTEGER*4  NTS(PIM__MFRQ), UTS(PIM__MFRQ), K_FRQ, IND_TSYS, IND_SCA, &
     &           IND_SOU, NPT, IFRQ, JFRQ, IND_TO_PT(PIM__MSCA), &
     &           KPOL, IF_REF, IND_TUS_STA
      INTEGER*4  J1, J2, J3, J4, J5, J6, J7, J8, J9, J10, J11, J12, IER
      LOGICAL*4, EXTERNAL :: IS_R8_NAN
      CHARACTER, EXTERNAL :: MJDSEC_TO_DATE*30, GET_CDATE*19
      INTEGER*4, EXTERNAL :: LTM_DIF
!
      K_FRQ = PIM%CONF%END_FRQ - PIM%CONF%BEG_FRQ + 1
!
! --- KPOL -- the number of polarization slots
!
      IF ( PIM%NPOL == 1 ) THEN
           KPOL = 1
         ELSE
           KPOL = 2
      END IF
!
! --- Allocate temporary arrays
!
      ALLOCATE ( TSVAL_IN(PIM%NSCA,K_FRQ), STAT=IER )
      IF ( IER .NE. 0 ) THEN
           CALL ERR_LOG ( 4121, IUER, 'PIMA_TSMO_IF', 'Error in an attempt '// &
     &         'to allocate dynamic memory for array TSVAL_IN' ) 
           RETURN 
      END IF
!
      ALLOCATE ( TSVAL_OUT(PIM%NSCA,K_FRQ), STAT=IER )
      IF ( IER .NE. 0 ) THEN
           CALL ERR_LOG ( 4122, IUER, 'PIMA_TSMO_IF', 'Error in an attempt '// &
     &         'to allocate dynamic memory for array TSVAL_OUT' ) 
           RETURN 
      END IF
!
      ALLOCATE ( TSFLG(PIM%NSCA,K_FRQ), STAT=IER )
      IF ( IER .NE. 0 ) THEN
           CALL ERR_LOG ( 4123, IUER, 'PIMA_TSMO_IF', 'Error in an attempt '// &
     &         'to allocate dynamic memory for array TSFLG' ) 
           RETURN 
      END IF
!
! --- Cycle over stations
!
      K_STA = 0
      DO 410 J1=1,PIM%NSTA
         IF ( .NOT. PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%AVAIL     ) THEN
              IF ( PIM%CONF%DEBUG_LEVEL .GE. 2 ) THEN
                   WRITE ( 6, 110 ) PIM%STA(J1)%IVS_NAME
 110               FORMAT ( 'No Tsys    for station ', A, ' skipped' )
              END IF
              GOTO 410
         END IF
!
         FL_TSYS_TOUSE = .TRUE.
         IND_TUS_STA = LTM_DIF ( 0, PIM%CONF%L_TUS, &
     &                              PIM%CONF%TSYS_USE_STA, PIM%C_STA(J1) )
         IF ( PIM%CONF%TUS_TYPE == PIMA__USE ) THEN
!
! ----------- The fine-grained list sets the list of station to use tsys
!
              IF ( IND_TUS_STA .LT. 1 ) FL_TSYS_TOUSE = .FALSE.
            ELSE IF ( PIM%CONF%TUS_TYPE == PIMA__NOT_USE ) THEN
!
! ----------- The fine-grained list sets the list of station not to use tsys
!
              IF ( IND_TUS_STA .GE. 1 ) FL_TSYS_TOUSE = .FALSE.
         END IF
!
         IF ( .NOT. FL_TSYS_TOUSE ) THEN
              IF ( PIM%CONF%DEBUG_LEVEL .GE. 2 ) THEN
                   WRITE ( 6, 120 ) PIM%STA(J1)%IVS_NAME
 120               FORMAT ( 'Tsys for station ', A, ' was deselected in the PIMA control file' )
              END IF
              GOTO 410
         END IF
!
         IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_OPA < 1       ) THEN
              IF ( PIM%CONF%DEBUG_LEVEL .GE. 3 ) THEN
                   WRITE ( 6, 130 ) PIM%STA(J1)%IVS_NAME
 130               FORMAT ( 'No opacity for station ', A, ' skipped' )
              END IF
              GOTO 410
         END IF
!
         NTS       = 0
         UTS       = 0
         TSVAL_IN  = -1.0
         TSVAL_OUT = -1.0
         NPT       = 0
         IND_TO_PT = 0
!
! ------ Cycle over epochs of scan  start time
!
         DO 420 J2=1,PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TAT-1,2
            IND_SCA = PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%IND_SCA(J2)
            IND_SOU = PIM%SCA(IND_SCA)%SOU_IND
!
! --------- Bypass observations with very low elevations
!
            IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%EL(J2)/DEG__TO__RAD < EL_LIM_MIN ) GOTO 420
!
! --------- Search for measured Tsys
!
            TIM_DIF_MIN = PIMA__SCM*PIM%CONF%MAX_SCAN_LEN
            TIM_STMO = ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TIM(J2) + &
     &                   PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TIM(J2+1) )/2.0D0
            IND_TSYS = 0
            DO 430 J3=1,PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%NPOI
               IF ( PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%SOU_IND(J3) == IND_SOU ) THEN
                    IF ( DABS(PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%TIME_MID_R8(J3) - TIM_STMO) < &
     &                   TIM_DIF_MIN ) THEN
!
                         TIM_DIF_MIN = DABS(PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%TIME_MID_R8(J3) - &
     &                                      TIM_STMO)
                         IND_TSYS = J3
                    END IF
               END IF
 430        CONTINUE 
!
! --------- Bypass points without measured Tsys
!
            IF ( IND_TSYS < 1 ) GOTO 420
!
! --------- Increment the counter of the array of measured Tsys and 
! --------- store in TSCAL_IN, TSCAL_OUT measured Tsys for all 
! --------- processed IFs within the band
!
            NPT = NPT + 1
            IFRQ = 0
            DO 440 J4=PIM%CONF%BEG_FRQ,PIM%CONF%END_FRQ
               IFRQ = IFRQ + 1
               TSYS_VAL = PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%TSYS(J4,IND_TSYS,IND_POL)
               IF ( IS_R8_NAN ( TSYS_VAL ) )  GOTO 440
               IF ( TSYS_VAL < TSYS_LIM_MIN ) GOTO 440
               IF ( TSYS_VAL > TSYS_LIM_MAX ) GOTO 440
               NTS(J4) = NTS(J4) + 1
               TSVAL_IN(NPT,IFRQ)  = TSYS_VAL
               TSVAL_OUT(NPT,IFRQ) = TSYS_VAL
 440        CONTINUE 
!
! --------- Update the cross reference array from scan start/stop epoch to the array of measured Tsys
!
            IND_TO_PT(J2)   = NPT
            IND_TO_PT(J2+1) = NPT
 420     CONTINUE 
!
! ------ Run the procedure for Tsys "if" modeling and compute array TSRAT:
! ------ Tsys ratios
!
         CALL ERR_PASS ( IUER, IER )
         CALL PIMA_TSYS_IF_ADJ ( PIM, J1, PIM%NSCA, K_FRQ, NPT, TSVAL_IN, IF_REF, TSFLG, &
     &                           TSRAT(1,J1), TSRMS(1,J1), IER )
         IF ( IER .NE. 0 ) THEN
              IF ( IER == 4131 ) THEN
                   IF ( PIM%CONF%CHECK_SEVERITY < 2 ) THEN
                        WRITE ( 6, * ) 'Nevertheless, continue' 
                        GOTO 410
                   END IF
              END IF
              CALL ERR_LOG ( 4124, IUER, 'PIMA_TSMO_IF', 'Error in an attempt '// &
     &            'to compute Tsys IF ratio for station '//PIM%STA(J1)%IVS_NAME )
              RETURN 
         END IF
         IF ( PIM%CONF%DEBUG_LEVEL .GE. 3 ) THEN
              WRITE ( 6, 140 ) PIM%STA(J1)%IVS_NAME, IF_REF + PIM%CONF%BEG_FRQ -1, &
     &                         TSRAT(1:K_FRQ,J1)
 140          FORMAT ( 'PIMA_TSMO_IF Sta: ', A, ' IF_Ref: ', I2, ' TSRat: ', 32(F7.4,1X) )
         END IF
!
! ------ Run a cycle over all start/stop epochs of scans of observations of the given station
!
         DO 450 J5=1,PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TAT
            IND_SCA = PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%IND_SCA(J5)
            IND_SOU = PIM%SCA(IND_SCA)%SOU_IND
            FL_TSYS_OK = .TRUE.
            IF ( IND_TO_PT(J5) > 0 ) THEN
                 IF ( .NOT. TSFLG(IND_TO_PT(J5),IF_REF) ) FL_TSYS_OK = .FALSE.
               ELSE 
                 FL_TSYS_OK = .FALSE.
            END IF
            TIM_STMO = PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TIM(J5)
            IF ( FL_TSYS_OK ) THEN
!
! -------------- Tsys for the reference frequency is OK
!
                 IFRQ = 0
                 DO 460 J6=PIM%CONF%BEG_FRQ,PIM%CONF%END_FRQ
                    IFRQ = IFRQ + 1
                    IF ( .NOT. TSFLG(IND_TO_PT(J5),IFRQ) ) THEN
!
! ---------------------- Tsys of the IFRQ-th frequency is flagged out.
! ---------------------- Replaced it with the Tsys for the reference frequency
! ---------------------- multiplied by the scaling factor
!
                         TSVAL_OUT(IND_TO_PT(J5),IFRQ) = TSRAT(IFRQ,J1)*TSVAL_IN(IND_TO_PT(J5),IF_REF)
                    END IF
 460             CONTINUE 
               ELSE
!
! -------------- Tsys for the reference frequency is bad
!
! -------------- Search for for the temporary reference frequency
!
                 IFRQ = 0
                 DO 470 J7=PIM%CONF%BEG_FRQ,PIM%CONF%END_FRQ
                    IFRQ = IFRQ + 1
                    FL_TSYS_OK = .TRUE.
                    IF ( IND_TO_PT(J5) > 0 ) THEN
                         IF ( .NOT. TSFLG(IND_TO_PT(J5),IFRQ) ) FL_TSYS_OK = .FALSE.
                       ELSE 
                         FL_TSYS_OK = .FALSE.
                    END IF
                    IF ( FL_TSYS_OK ) THEN
!
! ---------------------- Good! The J7th reference frequency is not flagged. 
! ---------------------- Set it the termporary reference
!
                         JFRQ = 0
                         DO 480 J8=PIM%CONF%BEG_FRQ,PIM%CONF%END_FRQ
                            JFRQ = JFRQ + 1
                            IF ( JFRQ == IFRQ ) GOTO 480
                            IF ( .NOT. TSFLG(IND_TO_PT(J5),JFRQ) ) THEN
!
! ------------------------------ The J8-th frequency is flagged out
! ------------------------------ Set Tsys for the J8th frequency equal to 
! ------------------------------ the Tsys of the J7-th frequency multiplied
! ------------------------------ by the scaling factor
!
                                 TSVAL_OUT(IND_TO_PT(J5),JFRQ) = TSRAT(JFRQ,J1)/TSRAT(IFRQ,J1)* &
     &                                                           TSVAL_IN(IND_TO_PT(J5),IFRQ)
                            END IF
 480                     CONTINUE 
                    END IF
 470             CONTINUE 
            END IF
 450     CONTINUE 
         IF ( PIM%CONF%DEBUG_LEVEL .EQ. 2 ) THEN
              WRITE ( 6, 150 ) PIM%STA(J1)%IVS_NAME
 150          FORMAT ( 'PIMA_TSMO_IF: processed station ', A)
         END IF 
         K_STA = K_STA + 1
         IF ( .NOT. ASSOCIATED ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN ) ) THEN
!
! ----------- Allocate memory for TSYS_CLN, since it was not allocated before
!
              PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TSYS = PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_OPA
              PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_AVAIL = .TRUE.
              ALLOCATE ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TSYS,PIM%NFRQ,KPOL), STAT=IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4125, IUER, 'PIMA_TSMO_IF', 'Error in an attempt '// &
     &                 'to allocate memory for object '// &
     &                 'PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN' )
                   RETURN 
              END IF
              PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN= -1.0D0
          END IF
!
         IF ( .NOT. ASSOCIATED ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD ) ) THEN
!
! ----------- Allocate memory for TSYS_MOD, since it was not allocated before
!
              ALLOCATE ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD(PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TSYS,PIM%NFRQ,KPOL), STAT=IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4126, IUER, 'PIMA_TSMO_IF', 'Error in an attempt '// &
     &                 'to allocate memory for object '// &
     &                 'PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD' )
                   RETURN 
              END IF
              PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD= -1.0D0
         END IF
!
! ------ Run a cycle over all start/stop epochs of scans of observations of the given station
! ------ and fill TSYS_MOD, TSYS_CLN
!
         DO 490 J9=1,PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TSYS
            IF ( IND_TO_PT(J9) < 1 ) GOTO 490
            STR = MJDSEC_TO_DATE ( PIM%MJD_0, PIM%TAI_0 + TIM_STMO, IER )
            IND_SCA = PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%IND_SCA(J9)
            IND_SOU = PIM%SCA(IND_SCA)%SOU_IND
            IFRQ = 0
            DO 4100 J10=PIM%CONF%BEG_FRQ,PIM%CONF%END_FRQ
               IFRQ = IFRQ + 1
               IF ( TSFLG(IND_TO_PT(J9),IFRQ) ) THEN
!
! ----------------- This point was not flagged. Keep observed Tsys
!
                    PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J9,J10,IND_POL) = TSVAL_IN(IND_TO_PT(J9),IFRQ)
                  ELSE
!
! ----------------- This point was flagged. Use modeld Tsys
!
                    PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J9,J10,IND_POL) = TSVAL_OUT(IND_TO_PT(J9),IFRQ)
               END IF
!
! ------------ Store modeled Tsys and Tsrat, the ratio of Tsys across IFs
!
               PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD(J9,J10,IND_POL) = TSVAL_OUT(IND_TO_PT(J9),IFRQ)
               PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT(J10,IND_POL)       = TSRAT(IFRQ,J1)
               PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT_AVAIL  = .TRUE.
               PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%IF_REF       = IF_REF
 4100       CONTINUE 
 490     CONTINUE 
 410  CONTINUE 
!
! --- Deallocate temporary memory
!
      DEALLOCATE ( TSFLG     )
      DEALLOCATE ( TSVAL_IN  )
      DEALLOCATE ( TSVAL_OUT )
      CALL ERR_LOG ( 0, IUER ) 
      RETURN
      END  SUBROUTINE  PIMA_TSMO_IF  !#!#
!
! ------------------------------------------------------------------------
!
      SUBROUTINE PIMA_TSYS_IF_ADJ ( PIM, ISTA, MPT, NFRQ, NPT, TSVAL, &
     &                              IF_REF, TSFLG, TSRAT, TSRMS, IUER )
! ************************************************************************
! *                                                                      *
! *   Routine PIMA_TSYS_IF_ADJ computes the ratio of Tsys over Ifs.      *
! *   PIMA_TSYS_IF_ADJ assumes the ratios are stable within the          *
! *   experiment. It tries all IFs as reference, computes the logarithms *
! *   of ratios, computes its median and run the iterative procedure     *
! *   for outliers rejection with respect to the median. It finds the    *
! *   reference IF that has less outliers. Then it computes the average  *
! *   ratios of Tsys at different IFs for the same observation with      *
! *   resepect to the reference IF and the rms defined as 
! *   It iteratively removes outliers that exceed rms*FRIB.NOISE_NSIGMA. *
! *   Results are written in STMO objects in arrays TSYS_MOD and         *
! *   TSYS_CLN.                                                          *
! *                                                                      *
! * ________________________ Input parameters: _________________________ *
! *                                                                      *
! *    PIM ( PIMA__TYP ) -- Object with information related to           *
! *                         program PIMA.                                *
! *   ISTA ( INTEGER*4 ) -- Station index.                               *
! *    MPT ( INTEGER*4 ) -- The maximum number of points in the input    *
! *                         arrays.                                      *
! *   NFRQ ( INTEGER*4 ) -- The number of IFs to be processed.           *
! *    NPT ( INTEGER*4 ) -- The number of points to be processed.        *
! *  TSVAL ( INTEGER*4 ) -- Input array of system temperatures.          *
! *                         Dimension: (MPT,NFRQ). Unit: K.              *
! *                                                                      *
! * ________________________ Output parameters: ________________________ *
! *                                                                      *
! *                                                                      *
! * IF_REF ( INTEGER*4 ) -- The index of reference IF.                   *
! *  TSFLG ( LOGICAL*1 ) -- Array of flags. .TRUE. means the Tsys        *
! *                         is considered good. .FALSE. means Tsys is    *
! *                         bad. Dimension: (MPT,NFRQ).                  *
! *  TSRAT ( REAL*8    ) -- Array of Tsys ratios across IFs.             *
! *                         TSRAT(I) = TSYS(x,I)/TSYS(x,IF_REF), where   *
! *                         x is the observation index.                  *
! *                         Dimension: NFRQ.                             *
! *  TSRMS ( REAL*8    ) -- Array of root mean squares of the Tsys       *
! *                         ratios after removing outleirs.              *
! *         TSRMS(I) = dexp ( rms of ln(Tsys(x,I)) - ln(Tsys(x,IF_REF)). *
! *                         Dimension: NFRQ.                             *
! *                                                                      *
! * ________________________ Modified parameters: ______________________ *
! *                                                                      *
! * IUER ( INTEGER*4, OPT ) -- Universal error handler.                  *
! *                      Input: IUER=0 -- no error message will be       *
! *                                       printed even in the case       *
! *                                       of error.                      *
! *                             IUER=-1,-2,-3 -- in the case of error    *
! *                                       the message will be put on     *
! *                                       stdout.                        *
! *                             IUER=-3 -- in the case of error after    *
! *                                        printing the error message    *
! *                                        the program will terminate.   *
! *                       Output:                                        *
! *                             if input value of IUER =-2,-3 -- IUER    *
! *                                        is not modified.              *
! *                             otherwise, the output value of IUER is 0 *
! *                             in the case of successful and            *
! *                             positive non-zero in the vase of errors. *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! * ### 26-AUG-2017 PIMA_TSYS_IF_ADJ v1.1 (d) L. Petrov  24-NOV-2020 ### *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      INCLUDE   'pima.i'
      TYPE     ( PIMA__TYPE  ) :: PIM
      INTEGER*4  MPT, NFRQ, NPT, IF_REF, IUER 
      REAL*8     TSVAL(MPT,NFRQ)
      LOGICAL*1  TSFLG(MPT,NFRQ)
      REAL*8     TSRAT(NFRQ), TSRMS(NFRQ)
      INTEGER*4  UPT_MIN
      PARAMETER  ( UPT_MIN = 5 )
      CHARACTER  STR1*1024, STR2*1024
      REAL*8     TSL(PIM__MSCA), TSL_MED, TSL_AVR(PIM__MFRQ,PIM__MFRQ), &
     &           TSL_RMS(PIM__MFRQ,PIM__MFRQ)
      REAL*8     TSYS_LIM_MIN, TSYS_LIM_MAX, TSL_RMS_FLOOR
      PARAMETER  ( TSYS_LIM_MIN  = 10.0D0 )
      PARAMETER  ( TSYS_LIM_MAX  = 950.0D0 )
      PARAMETER  ( TSL_RMS_FLOOR = 1.0D0 )
      INTEGER*4  J1, J2, J3, J4, J5, J6, J7, J8, J9, UPT, UPT_CLN, &
     &           UPT_ALL(PIM__MFRQ), NREJ, UPT_MAX, ISTA, IER 
      LOGICAL*4, EXTERNAL :: IS_R8_NAN
      INTEGER*4, EXTERNAL :: ILEN, I_LEN
!
      IF_REF  = 0
      UPT_MAX = 0
!
! --- Search for the reference frequency
!
      DO 410 J1=1,NFRQ
         UPT_ALL(J1) = 0
         DO 420 J2=1,NFRQ
            IF ( J2 == J1 ) GOTO 420
!
! --------- Make an array of Tsys of ratios of J2th frequency wrt the J1th frequency
!
            UPT = 0
            DO 430 J3=1,NPT
               IF ( TSVAL(J3,J1) < 0.0 .OR. TSVAL(J3,J2) < 0.0 ) GOTO 430
               UPT = UPT + 1
               TSL(UPT) = DLOG(TSVAL(J3,J2)/TSVAL(J3,J1))
 430        CONTINUE 
            IF ( UPT < UPT_MIN ) GOTO 420
!
! --------- Find the median TSL
!
            CALL SORT_R8  ( UPT, TSL )
            TSL_MED = TSL(UPT/2)
!
! --------- Compute the rms of the logarighm of Tsys ratio
!
            TSL_RMS(J2,J1) = 0.0D0
            DO 440 J4=1,UPT
               TSL_RMS(J2,J1)= TSL_RMS(J2,J1)+ (TSL_MED - TSL(J4))**2
 440        CONTINUE 
            TSL_RMS(J2,J1) = DSQRT ( TSL_RMS(J2,J1)/UPT + TSL_RMS_FLOOR**2 )
            TSL_AVR(J2,J1) = 0.0D0
            UPT_CLN = 0
            NREJ = 0
!
! --------- Now compute the average logarithm and reject outliers
!
            DO 450 J5=1,UPT
               IF ( DABS(TSL(J5) - TSL_MED) < PIM%CONF%FRIB_NOISE_NSIGMA*TSL_RMS(J2,J1) ) THEN
                    UPT_CLN = UPT_CLN + 1 ! Update the counter of good Tsys values
                    TSL_AVR(J2,J1) = TSL_AVR(J2,J1) + TSL(J5)
                 ELSE
                    NREJ = NREJ + 1
               END IF
 450        CONTINUE 
!
! --------- Compute the average logarithm 
!
            IF ( UPT_CLN > 1 ) THEN
                 TSL_AVR(J2,J1) = TSL_AVR(J2,J1)/UPT_CLN
            END IF 
!
! --------- Update the counter of the total number of good points
!
            UPT_ALL(J1) = UPT_ALL(J1) + UPT_CLN
 420     CONTINUE 
         IF ( UPT_ALL(J1) > UPT_MIN ) THEN
              IF ( UPT_ALL(J1) > UPT_MAX ) THEN
!
! ---------------- Select this frequency as the reference
!
                   IF_REF = J1
                   UPT_MAX = UPT_ALL(J1)
                   TSRAT(1:NFRQ) = DEXP(TSL_AVR(1:NFRQ,J1))
                   TSRMS(1:NFRQ) = DEXP(TSL_RMS(1:NFRQ,J1))
                   TSRAT(IF_REF) = 1.0D0
                   TSRMS(IF_REF) = 0.0D0
              END IF
         END IF
 410  CONTINUE 
      IF ( NFRQ == 1 ) THEN
           IF_REF   = 1
           TSRAT(1) = 1.0D0
           TSRMS(1) = 1.0D0
           TSRAT(1) = 1.0D0
           TSRMS(1) = 0.0D0
      END IF
      IF ( IF_REF == 0 ) THEN
           CALL ERR_PASS ( IUER, IER )
           CALL ERR_LOG  ( 4131, IER, 'PIMA_TSYS_IF_ADJ', 'Failed to find '// &
     &                    'the reference frequency for station '//PIM%C_STA(ISTA)// &
     &                    '. Skip this station' )
           CALL ERR_PASS ( 4131, IUER )
           RETURN 
      END IF
!
      DO 460 J6=1,NPT
         TSFLG(J6,1:NFRQ) = .FALSE.
         TSFLG(J6,IF_REF) = .TRUE.
         DO 470 J7=1,NFRQ
            IF ( J7 == IF_REF ) GOTO 470
            IF ( TSVAL(J6,J7) < TSYS_LIM_MIN .OR. TSVAL(J6,IF_REF) < TSYS_LIM_MIN ) GOTO 470
            TSL(J6) = DLOG(TSVAL(J6,J7)/TSVAL(J6,IF_REF))
!
            IF ( IS_R8_NAN ( TSRAT(J7) ) ) TSRAT(J7) = TSYS_LIM_MIN/10.0D0
            IF ( IS_R8_NAN ( TSRMS(J7) ) ) TSRMS(J7) = TSYS_LIM_MIN/10.0D0
            IF ( TSL(J6) - DLOG(TSRAT(J7)) > PIM%CONF%FRIB_NOISE_NSIGMA*DLOG(TSRMS(J7)) ) THEN
!
! -------------- Set flag "bad" for the J6th observation and J7th frequency
!
                 TSFLG(J6,J7) = .FALSE.
               ELSE IF ( TSL(J6) - DLOG(TSRAT(J7)) < - PIM%CONF%FRIB_NOISE_NSIGMA*DLOG(TSRMS(J7)) ) THEN
!
! -------------- We have the situation when the reference frequency is bad. Let us check
! -------------- other frequencies
!
                 TSFLG(J6,IF_REF) = .FALSE.
                 DO 480 J8=1,NFRQ-1
                    IF ( J8 == IF_REF ) GOTO 480
                    IF ( TSVAL(J6,J8) < 0.0 ) GOTO 480
!
! ----------------- The J8-th station becomes temporarily reference
!
                    DO 490 J9=J8+1,NFRQ
                       IF ( J9 == IF_REF ) GOTO 490
                       IF ( TSVAL(J6,J9) < 0.0 ) GOTO 490
                       IF ( DLOG(TSVAL(J6,J9)/TSVAL(J6,J8)) - DLOG(TSRAT(J9)/TSRAT(J8)) <  PIM%CONF%FRIB_NOISE_NSIGMA*DLOG(TSRMS(J9)) .AND. &
     &                      DLOG(TSVAL(J6,J9)/TSVAL(J6,J8)) - DLOG(TSRAT(J9)/TSRAT(J8)) > -PIM%CONF%FRIB_NOISE_NSIGMA*DLOG(TSRMS(J9))       ) THEN
!
! ------------------------- The ratios between J7th and J9th frequencies are good
!
                            TSFLG(J6,J8) = .TRUE.
                            TSFLG(J6,J9) = .TRUE.
                       END IF
 490                CONTINUE 
 480             CONTINUE 
               ELSE 
!
! -------------- Flag is OK for the J7th observation and the J6th frequency
!
                 IF ( TSFLG(J6,IF_REF) ) TSFLG(J6,J7) = .TRUE.
            END IF
 470     CONTINUE 
         IF ( PIM%CONF%DEBUG_LEVEL .GE. 6 ) THEN
              WRITE ( UNIT=STR1, FMT="(16(F6.1,' '))" ) TSVAL(J6,1:NFRQ)
              WRITE ( UNIT=STR2, FMT="(16(L1,' '))"   ) TSFLG(J6,1:NFRQ)
              WRITE ( 6, 210 ) PIM%STA(ISTA)%IVS_NAME, J6, STR1(1:I_LEN(STR1)), STR2(1:I_LEN(STR2))
 210          FORMAT ( 'PIMA_TSYS_IF_ADJ Sta: ', A, ' Pt: ', I4, ' Tsys: ', A, ' Flag: ', A )
              CALL FLUSH ( 6 )
         END IF
 460  CONTINUE 
!
      CALL ERR_LOG ( 0, IUER )
      RETURN
      END  SUBROUTINE  PIMA_TSYS_IF_ADJ   !#!#
!
! ------------------------------------------------------------------------
!
      SUBROUTINE PIMA_TSMO_ELEV ( PIM, K_STA, IND_POL, IUER )
! ************************************************************************
! *                                                                      *
! *   Routine PIMA_TSMO_ELEV decomposes measured system temperature as   *
! *                                                                      *
! *      T_sys = T_o * a(t) * b(e)                                       *
! *                                                                      *
! *   where  a(t) is a function of time represented by a linear spline;  *
! *          b(e) -- function of elevation represented by linear spline. *
! *          T_o  -- minimal system temperature.                         *
! *          Functions a(t) and b(e) are normalized to have minimal      *
! *          value 1.0                                                   *
! *                                                                      *
! *          Parameter T_o and coefficients of the spline a(t) and b(e)  *
! *   are found by iterative non-linear LSQ.                             *
! *                                                                      *
! *   PIMA_TSMO_ELEV runs the outliers elimination procedure.            *
! *   If Tsys record were procssed with tsmo "if" algorithm (stronly     *
! *   recommended), PIMA_TSMO_ELEV  takes as the input the cleaned Tsys  *
! *   that were not flagged out and averages them out using geometric    *
! *   averaging ( Av = (\Pi_i x_i)^^{1/n} ) after scaling to the         *
! *   the reference IF using TSRAT array.                                *
! *                                                                      *
! * ________________________ Input parameters: _________________________ *
! *                                                                      *
! * IND_POL ( INTEGER*4 ) -- Polarization index: 1 for R and 2 for L.    *
! *                                                                      *
! * ________________________ Output parameters: ________________________ *
! *                                                                      *
! *                                                                      *
! *   K_STA ( INTEGER*4 ) -- The number of processed station that have   *
! *                          input Tsys.                                 *
! *                                                                      *
! * ________________________ Modified parameters: ______________________ *
! *                                                                      *
! *       PIM ( PIMA__TYP ) -- Object with information related to        *
! *                            program PIMA.                             *
! * IUER ( INTEGER*4, OPT ) -- Universal error handler.                  *
! *                      Input: IUER=0 -- no error message will be       *
! *                                       printed even in the case       *
! *                                       of error.                      *
! *                             IUER=-1,-2,-3 -- in the case of error    *
! *                                       the message will be put on     *
! *                                       stdout.                        *
! *                             IUER=-3 -- in the case of error after    *
! *                                        printing the error message    *
! *                                        the program will terminate.   *
! *                       Output:                                        *
! *                             if input value of IUER =-2,-3 -- IUER    *
! *                                        is not modified.              *
! *                             otherwise, the output value of IUER is 0 *
! *                             in the case of successful and            *
! *                             positive non-zero in the vase of errors. *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! * ### 27-AUG-2017  PIMA_TSMO_ELEV  v1.2 (d) L. Petrov  05-DEC-2018 ### *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      INCLUDE   'astro_constants.i'
      INCLUDE   'pima.i'
      TYPE     ( PIMA__TYPE  ) :: PIM
      REAL*8       PIMA__SCM, EL_LIM_MIN, TSYS_LIM_MIN, TSYS_LIM_MAX 
      PARAMETER  ( PIMA__SCM = 1.25D0 )
      PARAMETER  ( EL_LIM_MIN   = 2.0D0*DEG__TO__RAD )
      PARAMETER  ( TSYS_LIM_MIN =  10.0D0 )
      PARAMETER  ( TSYS_LIM_MAX = 950.0D0 )
      REAL*8,    ALLOCATABLE :: TSVAL(:,:)
      LOGICAL*1, ALLOCATABLE :: TSFLG(:,:)
      CHARACTER  POLAR_STR*2
      REAL*8     TSYS_VAL, TIM_DIF_MIN, TSYS_ARR(PIM__MSCA), ELEV_ARR(PIM__MSCA), &
     &           TIM_ARR(PIM__MSCA)
      REAL*8     TIME_MOD(PIM__MSCA), TSYS_MOD(PIM__MSCA), TSYS_ZEN(PIM__MSCA), &
     &           TSYS_T0(PIM__MSCA), ELEV_T0(PIM__MSCA),  TIME_T(PIM__MSCA), &
     &           TSYS_T(PIM__MSCA), ELEV_E(PIM__MSCA), TSYS_E(PIM__MSCA), &
     &           TSYS_MOD_ALL(PIM__MSCA)
      REAL*8     TSYS_ZEN_MEAN, TSYS_RMS, TSYS_MIN, TSYS_MAX, TSYS_0
      INTEGER*4  IREF_INI(PIM__MSCA)
      LOGICAL*1  FLAG_INI(PIM__MSCA), FL_TSYS_TOUSE
      INTEGER*4  K_STA, IND_POL, IUER
      INTEGER*4  K_FRQ, IND_TSYS, IND_SCA, IND_SOU, NPT, IFRQ, JFRQ, &
     &           IND_TO_PT(PIM__MSCA), NTS, NTF, NM, NT, NE, NA
      INTEGER*4  J1, J2, J3, J4, J5, J6, J7, J8, J9, J10, J11, J12, &
     &           IND_TUS_STA, IER
      LOGICAL*4, EXTERNAL :: IS_R8_NAN
      INTEGER*4, EXTERNAL :: LTM_DIF
!
      K_FRQ = PIM%CONF%END_FRQ - PIM%CONF%BEG_FRQ + 1 ! The number of used IFs
!
! --- Cycle over stations
!
      K_STA = 0
      DO 410 J1=1,PIM%NSTA
         FL_TSYS_TOUSE = .TRUE.
         IND_TUS_STA = LTM_DIF ( 0, PIM%CONF%L_TUS, &
     &                              PIM%CONF%TSYS_USE_STA, PIM%C_STA(J1) )
         IF ( PIM%CONF%TUS_TYPE == PIMA__USE ) THEN
!
! ----------- The fine-grained list sets the list of station to use tsys
!
              IF ( IND_TUS_STA .LT. 1 ) FL_TSYS_TOUSE = .FALSE.
            ELSE IF ( PIM%CONF%TUS_TYPE == PIMA__NOT_USE ) THEN
!
! ----------- The fine-grained list sets the list of station not to use tsys
!
              IF ( IND_TUS_STA .GE. 1 ) FL_TSYS_TOUSE = .FALSE.
         END IF
!
         IF ( .NOT. FL_TSYS_TOUSE ) THEN
              IF ( PIM%CONF%DEBUG_LEVEL .GE. 2 ) THEN
                   WRITE ( 6, '(A)' ) 'PIMA_TSMO: station '//PIM%C_STA(J1)// &
     &                                ' is skipped because it was deselected '// &
     &                                'in the PIMA control file'
              END IF
              GOTO 410
         END IF
!
! ------ Bypass observations if Tsys is not available for this station
!
         IF ( .NOT. PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_AVAIL ) THEN
              IF ( PIM%CONF%DEBUG_LEVEL .GE. 1 ) THEN
                   WRITE ( 6, '(A)' ) 'PIMA_TSMO: station '//PIM%C_STA(J1)// &
     &                                ' is skipped because no opacity was available'
              END IF
              GOTO 410
         END IF
         NTS = 0
!
! ------ Cycle over epochs of scan start time
!
         DO 420 J2=1,PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TAT-1,2
            IND_SCA = PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%IND_SCA(J2)
            IND_SOU = PIM%SCA(IND_SCA)%SOU_IND
!
! --------- Bypass observations with very low elevations
!
            IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%EL(J2)/DEG__TO__RAD < EL_LIM_MIN ) GOTO 420
            TIM_DIF_MIN = PIMA__SCM*PIM%CONF%MAX_SCAN_LEN
            IND_TSYS = 0
!
! --------- Search for measured Tsys
!
            DO 430 J3=1,PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%NPOI
               IF ( PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%SOU_IND(J3) == IND_SOU ) THEN
                    IF ( DABS(PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%TIME_MID_R8(J3) - &
     &                        PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TIM(J2)) < TIM_DIF_MIN ) THEN
                         TIM_DIF_MIN = DABS(PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%TIME_MID_R8(J3) - &
     &                                 PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TIM(J2))
                         IND_TSYS = J3
                    END IF
               END IF
 430        CONTINUE 
!
! --------- Bypass points without measured Tsys
!
            IF ( IND_TSYS < 1 ) GOTO 420
!
! --------- Get Tsys value
!
            IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT_AVAIL .AND. &
     &           PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_AVAIL        ) THEN
!
! -------------- The Tsys records were processed with tsmo "if" alogirithm
!
                 NTF = 0
                 TSYS_VAL = 1.0D0
                 DO 440 J4=PIM%CONF%BEG_FRQ,PIM%CONF%END_FRQ
                    IF ( IS_R8_NAN ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J2,J4,IND_POL) ) ) THEN
                         PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J2,J4,IND_POL) = 0.0D0
                    END IF
                    IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J2,J4,IND_POL) > TSYS_LIM_MIN ) THEN
!
! ---------------------- Tsys is available
!
                         TSYS_VAL = TSYS_VAL* PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J2,J4,IND_POL)/ &
     &                                        PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT(J4,IND_POL)
                         NTF = NTF + 1
                    END IF
 440             CONTINUE 
                 IF ( NTF > 0 ) THEN
                      TSYS_VAL = TSYS_VAL**(1.0D0/NTF)
                 END IF
               ELSE
!
! -------------- Task TSMO was not executed. Well, then take measured Tsys at the first IF
!
                 TSYS_VAL = PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%TSYS(PIM%CONF%BEG_FRQ,IND_TSYS,IND_POL)
            END IF
!
! --------- Discard Tsys that are out of range
!
            IF ( TSYS_VAL < TSYS_LIM_MIN ) GOTO 420
            IF ( TSYS_VAL > TSYS_LIM_MAX ) GOTO 420
!
! --------- Put Tsys, elevation and time in arrays TSYS_ARR, ELEV_ARR and TIM_ARR.
! --------- And, by the way, update IND_TO_PT array of cross-references
!
            NTS = NTS + 1
            IND_TO_PT(J2)   = NTS
            IND_TO_PT(J2+1) = NTS
            TSYS_ARR(NTS) = TSYS_VAL
            ELEV_ARR(NTS) = PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%EL(J2) 
            TIM_ARR(NTS)  = PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TIM(J2) 
 420     CONTINUE 
         IF ( PIM%CONF%DEBUG_LEVEL .GE. 6 ) THEN
              WRITE ( 6, * ) 'PIMA_TSMO_ELEV Sta: ', PIM%STA(J1)%IVS_NAME, &
     &                       ' PTE: ', PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TSYS, ' NTS = ', NTS
              IF ( NTS > 0 ) THEN
                   WRITE ( 6, * ) 'TIM_ARR= ', TIM_ARR(1), TIM_ARR(NTS), &
     &                            ' TSRAT= ', SNGL(PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT(PIM%CONF%BEG_FRQ:PIM%CONF%END_FRQ,IND_POL))
              END IF
         END IF
         IF ( NTS == 0 ) THEN
              IF ( PIM%CONF%DEBUG_LEVEL .GE. 1 ) THEN
                   WRITE ( 6, * ) 'PIMA_TSMO: station '//PIM%C_STA(J1)// &
     &                            ' is skipped since no valid Tsys measurements were found'
              END IF
         END IF
!
! ------ Run the procesdure of Tsys decomposution in to product of three constutents:
! ------ T_sys = T_o * a(t) * b(e)
!
         CALL ERR_PASS ( IUER, IER )
         CALL TSYS_MODEL ( NTS,  TIM_ARR, ELEV_ARR, TSYS_ARR, &
     &                     NM, TIME_MOD,   TSYS_MOD, TSYS_ZEN, TSYS_T0, ELEV_T0, &
     &                     NT, TIME_T,     TSYS_T, &
     &                     NE, ELEV_E,     TSYS_E, &
     &                     PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TAT, &
     &                     PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TIM,   &
     &                     PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%EL,    &
     &                     TSYS_MOD_ALL, &
     &                     IREF_INI, FLAG_INI, TSYS_ZEN_MEAN, TSYS_RMS, &
     &                     TSYS_MIN, TSYS_MAX, TSYS_0, PIM%CONF%DEBUG_LEVEL, IER )
         IF ( IER .NE. 0 ) THEN
              CALL ERR_LOG ( 4141, IUER, 'PIMA_TSMO_ELEV', 'Error in '// &
     &            'computing the Tsys model of decomposition it to '// &
     &            'a product of time dependent and elevation dependent '// &
     &            ' splines for station '//PIM%C_STA(J1) )
              RETURN 
         END IF
         IF ( NM == 0 .AND. NT == 0 .AND. NE == 0 ) THEN
              WRITE ( 6, '(A)' ) 'Task TSMO failed for station '//PIM%C_STA(J1)// &
     &                           ': too few points for Tsys model computation. '
              WRITE ( 6, '(A)' ) 'Please, try to increase spline span by using '
              WRITE ( 6, '(A)' ) 'PIMAVAR_TSYS_SPAN_T envorinment variable'
              GOTO 410
!!              CALL EXIT ( 1 )
         END IF
         IF ( PIM%CONF%DEBUG_LEVEL .GE. 8 ) THEN
!
! ----------- Make plots if PIMA_TSMO was called in th dbugging mode
!
              WRITE ( 6, * ) 'PIMA-TSMO 916 NE= ', NE,  ' NM= ', NM, ' N_TAT= ', PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TAT
              IF ( NE > 0 .AND. NM > 0 .AND. PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TAT > 0 ) THEN
                   CALL DIAGI_SETDEF ( IER, 'DIAGI_CTIT', 'Tsys at zenith for '//PIM%STA(J1)%IVS_NAME )
                   CALL DIAGI_2 ( NT, TIME_T, TSYS_T, NM, TIME_MOD, TSYS_ZEN, IER )
                   CALL DIAGI_SETDEF ( IER, 'DIAGI_CTIT', 'Tsys elev dependence for '//PIM%STA(J1)%IVS_NAME )
                   CALL FLUSH ( 6 )
                   CALL DIAGI_3 ( NE, ELEV_E, TSYS_E, &
     &                            NM, ELEV_T0, TSYS_T0, &
     &                            PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TAT, &
     &                            PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%EL,   &
     &                            TSYS_MOD_ALL, IER )
                   CALL DIAGI_SETDEF ( IER, 'DIAGI_CTIT', 'Tsys mod '//pim%sta(j1)%ivs_name )
                   CALL DIAGI_3 ( NM, TIME_MOD, TSYS_MOD, &
     &                            NTS, TIM_ARR, TSYS_ARR, &
     &                            PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TAT, &
     &                            PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TIM,   &
     &                            TSYS_MOD_ALL, IER )
              END IF
         END IF
!
! ------ Now update arrays of cleaned and modeled Tsys:
!
         DO 450 J5=1,PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%N_TAT-1,2 ! Again, cycle over epochs of scan start time
            IND_SCA = PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%IND_SCA(J5)
            IND_SOU = PIM%SCA(IND_SCA)%SOU_IND
!
! --------- Discard the observation with elevation angle below the limit
!
            IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%EL(J5)/DEG__TO__RAD < EL_LIM_MIN ) GOTO 450
            TIM_DIF_MIN = PIMA__SCM*PIM%CONF%MAX_SCAN_LEN
!
! --------- Find the index to the measured Tsys
!
            IND_TSYS = 0
            DO 460 J6=1,PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%NPOI
               IF ( PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%SOU_IND(J6) == IND_SOU ) THEN
                    IF ( DABS(PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%TIME_MID_R8(J6) - &
     &                        PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TIM(J5)) < TIM_DIF_MIN ) THEN
                         TIM_DIF_MIN = DABS(PIM%STA(J1)%TSYS(PIM%CONF%FRQ_GRP)%TIME_MID_R8(J6) - &
     &                                 PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TIM(J5))
                         IND_TSYS = J6
                    END IF
               END IF
 460        CONTINUE 
            IF ( IND_TSYS > 0 .AND. ASSOCIATED ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD ) ) THEN
!
! -------------- Tsys was measured for this observation
!
                 DO 470 J7=PIM%CONF%BEG_FRQ,PIM%CONF%END_FRQ
                    IF ( IS_R8_NAN ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD(J5,J7,IND_POL) ) ) THEN
                         PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD(J5,J7,IND_POL) = 0.0D0
                    END IF
                    PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD(J5,J7,IND_POL) = &
     &                           TSYS_MOD_ALL(J5)*PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT(J7,IND_POL)
                    IF ( IND_TO_PT(J5) > 0 ) THEN
                         IF ( .NOT. FLAG_INI(IND_TO_PT(J5)) ) THEN
                               IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT_AVAIL ) THEN
!
! --------------------------------- The Tsys was measured for this observation, 
! --------------------------------- and it was was flgagged out for this IF.
! --------------------------------- Then replace the measured Tsys with the modeled Tsys 
! --------------------------------- fot this element of the model arry
!
                                    PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J7,IND_POL) = &
     &                                  TSYS_MOD_ALL(J5)*PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT(J7,IND_POL)
                                  ELSE
                                    PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J7,IND_POL) = TSYS_MOD_ALL(J5)
                               END IF
                          END IF
                    END IF
                    IF ( IS_R8_NAN ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J7,IND_POL)  ) ) THEN
                         PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J7,IND_POL) = 0.0D0
                    END IF
                    IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J7,IND_POL) < TSYS_LIM_MIN ) THEN
!
! ---------------------- There was not good Tsys measurement for this observation, this IF
!
                         IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT_AVAIL ) THEN
                              PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J7,IND_POL) = &
     &                                    TSYS_MOD_ALL(J5)*PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT(J7,IND_POL)
                            ELSE
                              PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J7,IND_POL) = TSYS_MOD_ALL(J5)
                         END IF
                    END IF
!
! ----------------- Update modeled and cleaned Tsys for the epoch of the end of scan
!
                    PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5+1,J7,IND_POL) = &
     &                          PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J7,IND_POL) 
                    PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD(J5+1,J7,IND_POL) = &
     &                          PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD(J5,J7,IND_POL) 
 470             CONTINUE
               ELSE
!
! -------------- Tsys was not measured for this observation
!
                 IF ( ASSOCIATED ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN ) ) THEN
                      DO 480 J8=PIM%CONF%BEG_FRQ,PIM%CONF%END_FRQ
                         IF ( IS_R8_NAN ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J8,IND_POL) ) ) THEN
                              PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J8,IND_POL) = 0.0D0
                         END IF
                         IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J8,IND_POL) < TSYS_LIM_MIN ) THEN
!
! ---------------------------- Cleaned Tsys is flagged out. We need update it
!
                               IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT_AVAIL ) THEN
!
! --------------------------------- TSRAT was availble: use it
!
                                    PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J8,IND_POL) = &
     &                                      TSYS_MOD_ALL(J5)*PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT(J8,IND_POL)
                                  ELSE
!
! -------------------------------- TSRAT was not availble. Take tsys for the model value for 
! -------------------------------- "some" IF.
!
                                   PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J8,IND_POL) = &
     &                                     TSYS_MOD_ALL(J5)
                               END IF
                         END IF
!
! ---------------------- Update modeled Tsys
!
                         IF ( PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT_AVAIL ) THEN
                              PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD(J5,J8,IND_POL) = &
     &                                    TSYS_MOD_ALL(J5)*PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSRAT(J8,IND_POL)
                            ELSE
                              PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD(J5,J8,IND_POL) = &
     &                                    TSYS_MOD_ALL(J5)
                         END IF
!
! ---------------------- Update modeled and cleaned Tsys for the epoch of the end of scan
!
                         PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5+1,J8,IND_POL) = &
     &                               PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_CLN(J5,J8,IND_POL) 
                         PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD(J5+1,J8,IND_POL) = &
     &                               PIM%STA(J1)%STMO(PIM%CONF%FRQ_GRP)%TSYS_MOD(J5,J8,IND_POL) 
 480                  CONTINUE 
                 END IF
            END IF
 450     CONTINUE 
         K_STA = K_STA + 1
 410  CONTINUE 
      CALL ERR_LOG ( 0, IUER )
      RETURN
      END  SUBROUTINE  PIMA_TSMO_ELEV  !#!  
