      SUBROUTINE GET_FITS_MAP ( FINAM, FL_CC, FL_IMA, MAP, IUER )
! ************************************************************************
! *                                                                      *
! *   Routine  GET_FITS_MAP reads the radio source map in fits format    *
! *   generated by programs AIPS or DIFMAP and fills fields of the       *
! *   object MAP. Map fits file contains map in the form of the          *
! *   two-dimensional pixel array and in the form of a sum of delta      *
! *   functions and the synthesized beam which represent the image --    *
! *   so-called Clean components. Routine GET_FITS_MAP can get the image *
! *   in eigther of these form, or both or none. If the object MAP       *
! *   contained the image before call of GET_FITS_MAP, its content is    *
! *   lost.                                                              *
! *                                                                      *
! * _________________________ Input parameters: ________________________ *
! *                                                                      *
! *  FINAM ( CHARACTER ) -- Name of the input FITS file with map.        *
! *  FL_CC ( LOGICAL*4 ) -- Flag, whether or not to read the image in    *
! *                         the form of a sum of delta functions.        *
! * FL_IMA ( LOGICAL*4 ) -- Flag, whether or not to read the image in    *
! *                         the form of a two-dimesional pixel array.    *
! *                i                                                     *
! * _________________________ Output parameters: _______________________ *
! *                                                                      *
! *    MAP ( SOUMAP__TYPE ) -- object that keeps variables which         *
! *                            describe the image.                       *
! *                                                                      *
! * ________________________ Modified parameters: ______________________ *
! *                                                                      *
! *    IUER ( INTEGER*4, OPT ) -- Universal error handler.               *
! *                      Input: IUER=0 -- no error message will be       *
! *                                       printed even in the case       *
! *                                       of error.                      *
! *                             IUER=-1,-2,-3 -- in the case of error    *
! *                                       the message will be put on     *
! *                                       stdout.                        *
! *                             IUER=-3 -- in the case of error after    *
! *                                        printing the error message    *
! *                                        the program will terminate.   *
! *                       Output:                                        *
! *                             if input value of IUER =-2,-3 -- IUER    *
! *                                        is not modified.              *
! *                             otherwise, the output value of IUER is 0 *
! *                             in the case of successful and            *
! *                             positive non-zero in the case of errors. *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ## 26-FEB-2004   GET_FITS_MAP  v1.3 (d)  L. Petrov  03-SEP-2018 ### *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      INCLUDE   'astro_constants.i'
      INCLUDE   'sou_map.i'
      TYPE     ( SOUMAP__TYPE ) :: MAP
      CHARACTER  FINAM*(*)
      LOGICAL*4  FL_CC, FL_IMA
      INTEGER*4  IUER
      CHARACTER  COMMENT*128, DATE_CHR*21, STR*32
      CHARACTER, ALLOCATABLE :: KEYS(:,:)*80
      LOGICAL*4  ANYF, FL_FLUX, FL_X, FL_Y
      REAL*4     NULLVAL
      REAL*4,    ALLOCATABLE :: ARR_R4(:)
      INTEGER*8  FPTR
      INTEGER*4  BITPIX, INUM, HDU_TYPE, GROUP, J1, J2, J3, J4, J5, &
     &           IND_COL, LKEY(MHDR), LHDR, FT_STATUS, IP, KH, IVRB, IER
      INTEGER*4, EXTERNAL :: ILEN, I_LEN, CUNIT2FITS 
!
      ALLOCATE ( KEYS(MKEY,MHDR), STAT=IER )
      IF ( IER .NE. 0 ) THEN
           CALL CLRCH ( STR )
           CALL IINCH ( MKEY*MHDR*LEN(KEYS(1,1)) )
           CALL ERR_LOG ( 4811, IUER, 'GET_FITS_MAP', 'Failure to allocate '// &
     &          STR(1:I_LEN(STR))//' bytes memory for array KEYS' ) 
           RETURN
      END IF
!
      FT_STATUS = 0
      IVRB = 0
      KH = 1
!
      CALL GETENVAR ( 'FITS_MAP_IMAGE_TABLE', STR )
      IF ( ILEN(STR) > 0 ) THEN
           CALL CHIN ( STR, KH )
      END IF
!
! --- Release dynamic memory which may have been allocated eariler
!
      CALL MAP_FREE ( MAP )
!
! --- Open the fits file
!
      CALL ERR_PASS ( IUER, IER ) 
      CALL FFITS_OPEN ( FINAM, FPTR, 'OLD', IER )
      IF ( IER .NE. 0 ) THEN
           CALL ERR_LOG ( 4812, IUER, 'GET_FITS_MAP', 'Error in opening '// &
     &                   'fits file '//FINAM )
           DEALLOCATE ( KEYS )
           RETURN
      END IF
!
! --- Read the headers and get the array of keys
!
      CALL ERR_PASS ( IUER, IER ) 
      CALL FFITS_GET_KEYS ( FPTR, MHDR, MKEY, LHDR, LKEY, KEYS, IER )
      IF ( IER .NE. 0 ) THEN
           CALL ERR_LOG ( 4813, IUER, 'GET_FITS_MAP', 'Error in attempt '// &
     &         'to read the header of fits file '//FINAM )
           CALL FFITS_CLOSE ( FPTR, -2 )
           DEALLOCATE ( KEYS )
           RETURN
      END IF
!
      IF ( KH .NE. 1 ) THEN
!
! -------- Position to the 2nd header unit
!
           CALL FFMAHD ( %VAL(FPTR), %VAL(KH), HDU_TYPE, FT_STATUS )
           IF ( FT_STATUS .NE. 0 ) THEN
                CALL FT_PRINTERROR ( 4826, IUER, 'GET_FITS_MAP', FT_STATUS )
                CALL ERR_LOG ( 4827, IUER, 'GET_FITS_MAP', 'Error in '// &
     &              'an attempt to move to the second header unit of '// &
     &              'the fits file '//FINAM )
                CALL FFITS_CLOSE ( FPTR, -2 )
           END IF
      END IF
!
! --- Scanning all keyword records for something useful
!
      DO 410 J1=1,LKEY(KH)
         IF ( KEYS(J1,KH)(1:8) == 'NAXIS1  ' ) THEN
!
! ----------- Get the first image dimension
!
              READ ( UNIT=KEYS(J1,KH)(21:30), FMT='(I10)', IOSTAT=IER ) MAP%DIM1
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4814, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the first dimension of the map '// &
     &                 'from the fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
         END IF
!
         IF ( KEYS(J1,KH)(1:8) == 'NAXIS2  ' ) THEN
!
! ----------- Get the second image dimension
!
              READ ( UNIT=KEYS(J1,KH)(21:30), FMT='(I10)', IOSTAT=IER ) MAP%DIM2
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4815, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the first dimension of the map '// &
     &                 'from the fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
         END IF
!
         IF ( ( KH == 1 .AND. KEYS(J1,KH)(1:8) == 'CDELT1  ' ) .OR. KEYS(J1,KH)(1:8) == 'CD1_1   ' ) THEN
!
! ----------- Get the pixel step in right ascension
!
              READ ( UNIT=KEYS(J1,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%STEP_RA
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4816, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the right ascension pixel '// &
     &                 'decrement of the map from the fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
              MAP%STEP_RA = MAP%STEP_RA * DEG__TO__RAD
         END IF
!
         IF ( KEYS(J1,KH)(1:8) == 'CRVAL1  ' ) THEN
!
! ----------- Get the right ascension of the image phase center
!
              READ ( UNIT=KEYS(J1,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%ALPHA
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4817, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the the right ascension '// &
     &                 'of the reference pixel of the map from the '// &
     &                 'fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
              MAP%ALPHA = MAP%ALPHA * DEG__TO__RAD
         END IF
!
         IF ( ( KH == 1 .AND. KEYS(J1,KH)(1:8) == 'CDELT2  ' ) .OR. KEYS(J1,KH)(1:8) == 'CD2_2   ' ) THEN
!
! ----------- Get the pixel step in right declination 
!
              READ ( UNIT=KEYS(J1,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%STEP_DL 
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4818, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the declination pixel '// &
     &                 'decrement of the map from the fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
              MAP%STEP_DL = MAP%STEP_DL * DEG__TO__RAD
         END IF
!
         IF ( KEYS(J1,KH)(1:8) == 'CRVAL2  ' ) THEN
!
! ----------- Get the declination of the image phase center
!
              READ ( UNIT=KEYS(J1,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%DELTA
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4819, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the the declination '// &
     &                 'of the reference pixel of the map from the '// &
     &                 'fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
              MAP%DELTA = MAP%DELTA * DEG__TO__RAD
         END IF
!
         IF ( KEYS(J1,KH)(11:20) == "'FREQ    '" ) THEN
!
! ----------- Get the reference frequency of the image
!
              IF ( KEYS(J1+1,KH)(1:5) == 'CRVAL' ) THEN
                   READ  ( UNIT=KEYS(J1+1,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%FREQ
                 ELSE IF ( KEYS(J1+2,KH)(1:5) == 'CRVAL' ) THEN
                   READ  ( UNIT=KEYS(J1+2,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%FREQ
                 ELSE IF ( KEYS(J1+3,KH)(1:5) == 'CRVAL' ) THEN
                   READ  ( UNIT=KEYS(J1+3,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%FREQ
              END IF
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4820, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the sky frequency of '// &
     &                 'the map from the fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
         END IF
!
         IF ( KEYS(J1,KH)(1:8) == 'OBJECT  ' ) THEN
!
! ----------- Get the name of the object 
!
              MAP%SOU_NAME = KEYS(J1,KH)(12:21)
              IP = INDEX ( MAP%SOU_NAME, "'" )
              IF ( IP > 0 ) CALL CLRCH ( MAP%SOU_NAME(IP:) )
         END IF
         IF ( KEYS(J1,KH)(1:8) == 'OBSERVER' ) THEN
!
! ----------- Get the name of the object 
!
              MAP%EXP_NAME = KEYS(J1,KH)(12:23)
              IP = INDEX ( MAP%EXP_NAME, "'" )
              IF ( IP > 0 ) CALL CLRCH ( MAP%EXP_NAME(IP:) )
         END IF
         IF ( KEYS(J1,KH)(1:8) == 'DATE-OBS' ) THEN
!
! ----------- Get the observation date
!
              MAP%DATE_OBS= KEYS(J1,KH)(12:21)
              IF ( MAP%DATE_OBS(5:5) == '-' ) THEN
!
! ---------------- Transform the date to VTD format
!
                   MAP%DATE_OBS = MAP%DATE_OBS(1:4)//'.'// &
     &                            MAP%DATE_OBS(6:7)//'.'// &
     &                            MAP%DATE_OBS(9:10)
              END IF
         END IF
!
         IF ( KEYS(J1,KH)(1:8) == 'DATAMAX ' ) THEN
!
! ----------- Get the brightness peak maximum
!
              READ ( UNIT=KEYS(J1,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%FLUX_MAX
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4821, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the sky map maxumim flux density '// &
     &                 'from the fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
         END IF
!
         IF ( KEYS(J1,KH)(1:8) == 'BMAJ    ' ) THEN
!
! ----------- Get the semi-major axis of the ellipsoid that approximates the 
! ----------- synthesized beam
!
              READ ( UNIT=KEYS(J1,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%BEAM_MAJ
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4822, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the semi-major axis of the beam '// &
     &                 'of the map from the fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
              MAP%BEAM_MAJ = MAP%BEAM_MAJ*DEG__TO__RAD
         END IF
!
         IF ( KEYS(J1,KH)(1:8) == 'BMIN    ' ) THEN
!
! ----------- Get the semi-minor axis of the ellipsoid that approximates the 
! ----------- synthesized beam
!
              READ ( UNIT=KEYS(J1,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%BEAM_MIN
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4823, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the semi-minor axis of the beam '// &
     &                 'of the map from the fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
              MAP%BEAM_MIN = MAP%BEAM_MIN*DEG__TO__RAD
         END IF
!
         IF ( KEYS(J1,KH)(1:8) == 'BPA     ' ) THEN
!
! ----------- Get the position angle of the ellipsoid that approximates the 
! ----------- synthesized beam
!
              READ ( UNIT=KEYS(J1,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%BEAM_POS_ANG
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4824, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the position angles of axis '// &
     &                 'of the beam of the map from the fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
              MAP%BEAM_POS_ANG = MAP%BEAM_POS_ANG*DEG__TO__RAD
         END IF
!
         IF ( KEYS(J1,KH)(1:8) == 'IM_NOISE'.OR. KEYS(J1,KH)(1:8) == 'IMNOISE ' ) THEN
!
! ----------- Get the rms of the image noise 
!
              READ ( UNIT=KEYS(J1,KH)(11:30), FMT='(F20.10)', IOSTAT=IER ) MAP%NOISE
              IF ( IER .NE. 0 ) THEN
                   WRITE ( 6, * ) 'KEYS(J1,KH)(11:30) >>',KEYS(J1,KH)(11:30),'<<'
                   CALL ERR_LOG ( 4825, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'attempt to decode the noise level of the map from '// &
     &                 'the fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
         END IF
!
         IF ( KEYS(J1,KH)(1:8) == 'DATE-OBS' ) THEN
!
! ----------- Get the observation date
!
              CALL CLRCH ( DATE_CHR )
              IF ( KEYS(J1,KH)(14:14) == '/' ) THEN
                   DATE_CHR = '19'//KEYS(J1,KH)(18:19)//'_'// &
     &                              KEYS(J1,KH)(15:16)//'_'// &
     &                              KEYS(J1,KH)(12:13)//'_00:00:00.0'
                   DATE_CHR(5:5) = '_'
                   DATE_CHR(8:8) = '_'
                 ELSE 
                   DATE_CHR = KEYS(J1,KH)(12:21)//'_00:00:00.0'
                   DATE_CHR(5:5) = '_'
                   DATE_CHR(8:8) = '_'
              END IF
              IF ( KEYS(J1,KH)(22:22) == '(' ) THEN
                   IP = INDEX ( KEYS(J1,KH)(22:), ')' ) + 21
                   IF ( IP > 23 ) THEN
                        CALL CHIN ( KEYS(J1,KH)(23:IP-1), MAP%NUM_SEG )
                      ELSE
                        MAP%NUM_SEG = 1
                   END IF
                 ELSE 
                   MAP%NUM_SEG = 1
              END IF
!
              IER = -1
              CALL DATE_TO_TIME ( DATE_CHR, MAP%MJD, MAP%TAI, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 4826, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                 'an attempt to decode the image date from the '// &
     &                 'fits file '//FINAM )
                   CALL FFITS_CLOSE ( FPTR, -2 )
              END IF
         END IF
 410  CONTINUE 
      IF ( FL_CC ) THEN
!
! -------- Position to the 2nd header unit
!
           CALL FFMAHD ( %VAL(FPTR), %VAL(2), HDU_TYPE, FT_STATUS )
           IF ( FT_STATUS .NE. 0 ) THEN
                CALL FT_PRINTERROR ( 4826, IUER, 'GET_FITS_MAP', FT_STATUS )
                CALL ERR_LOG ( 4827, IUER, 'GET_FITS_MAP', 'Error in '// &
     &              'an attempt to move to the second header unit of '// &
     &              'the fits file '//FINAM )
                CALL FFITS_CLOSE ( FPTR, -2 )
           END IF
!
! -------- Scan the keywords in search of the number of clean components
!
           MAP%NUM_CC = 0
           DO 420 J2=1,LKEY(2)
              IF ( KEYS(J2,2)(1:8) == 'NAXIS2  ' ) THEN
                   READ ( UNIT=KEYS(J2,2)(21:30), FMT='(I10)', IOSTAT=IER ) MAP%NUM_CC
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 4828, IUER, 'GET_FITS_MAP', 'Error in '// &
     &                      'attempt to decode the number of clean components '// &
     &                      'from the fits file '//FINAM )
                        CALL FFITS_CLOSE ( FPTR, -2 )
                   END IF
              END IF
 420       CONTINUE 
!
! ======== Read the image in the form of a sum of delta functions &
! ======== (Clean components)
!
! -------- Allocate dynamic memory
!
           ALLOCATE ( MAP%FLUX_CC(MAP%NUM_CC), STAT=IER )
           IF ( IER .NE. 0 ) THEN
                CALL CLRCH ( STR )
                CALL IINCH ( 4*MAP%NUM_CC, STR )
                CALL ERR_LOG ( 4829, IUER, 'GET_FITS_MAP', 'Failure to '// &
     &              'allocate '//STR(1:I_LEN(STR))//' bytes of dynamic '// &
     &              'memory for the array of fluxes of clean components' )
                DEALLOCATE ( KEYS )
                RETURN 
           END IF
!
           ALLOCATE ( MAP%COOR_CC(2,MAP%NUM_CC), STAT=IER )
           IF ( IER .NE. 0 ) THEN
                CALL CLRCH ( STR )
                CALL IINCH ( 4*2*MAP%NUM_CC, STR )
                CALL ERR_LOG ( 4830, IUER, 'GET_FITS_MAP', 'Failure to '// &
     &              'allocate '//STR(1:I_LEN(STR))//' bytes of dynamic '// &
     &              'memory for the array of coordinates, of clean components' )
                DEALLOCATE ( KEYS )
                RETURN 
           END IF
!
           ALLOCATE ( ARR_R4(MAP%NUM_CC), STAT=IER )
           IF ( IER .NE. 0 ) THEN
                CALL CLRCH ( STR )
                CALL IINCH ( 4*MAP%NUM_CC, STR )
                CALL ERR_LOG ( 4831, IUER, 'GET_FITS_MAP', 'Failure to '// &
     &              'allocate '//STR(1:I_LEN(STR))//' bytes of dynamic '// &
     &              'memory the temporary array' )
                DEALLOCATE ( KEYS )
                RETURN 
           END IF
!
           MAP%STATUS_CC = SMP__ALLC
!
           FT_STATUS = 0
!
           FL_FLUX = .FALSE.
           FL_X    = .FALSE.
           FL_Y    = .FALSE.
!
! -------- Again, scan the keys
!
           DO 430 J3=1,LKEY(2)
              IF ( KEYS(J3,2)(12:19) == 'FLUX    ' ) THEN
!
! ---------------- Found the keyword FLUX. Good sign. Get the column index
!
                   CALL CHIN ( KEYS(J3,2)(6:8), IND_COL )
                   IF ( IND_COL < 1 .OR. IND_COL > 999 ) THEN
                        CALL ERR_LOG ( 4832, IUER, 'GET_FITS_MAP', 'Failure to '// &
     &                      'decode TYYPE part of the FLUX keyword of the fits '// &
     &                      'file '//FINAM )
                        DEALLOCATE ( KEYS )
                        RETURN 
                   END IF
!
! ---------------- Get arrayu of amplitudes of clean components
!
                   FT_STATUS = 0 
                   CALL FFGCVE ( %VAL(FPTR), %VAL(IND_COL), %VAL(INT8(1)), &
     &                           %VAL(INT8(1)), %VAL(INT8(MAP%NUM_CC)), &
     &                           %VAL(1.0), MAP%FLUX_CC, ANYF, FT_STATUS )
                   IF ( FT_STATUS .NE. 0 ) THEN
                        CALL FT_PRINTERROR ( 4832, IUER, 'GET_FITS_MAP', &
     &                                       FT_STATUS )
                        CALL ERR_LOG ( 4833, IUER, 'GET_FITS_MAP', 'Failure to '// &
     &                      'get array of fluxes of the clean components from '// &
     &                      'the fits file '//FINAM )
                        DEALLOCATE ( KEYS )
                        RETURN 
                   END IF
                   FL_FLUX = .TRUE.
                ELSE IF ( KEYS(J3,2)(12:19) == 'DELTAX  ' ) THEN
!
! ---------------- Found the keyword DELTAX. Get the column index
!
                   CALL CHIN ( KEYS(J3,2)(6:8), IND_COL )
                   IF ( IND_COL < 1 .OR. IND_COL > 999 ) THEN
                        CALL ERR_LOG ( 4834, IUER, 'GET_FITS_MAP', 'Failure to '// &
     &                      'decode TYYPE part of the FLUX keyword of the fits '// &
     &                      'file '//FINAM )
                        DEALLOCATE ( KEYS )
                        RETURN 
                   END IF
!
! ---------------- Get array of U-coordinates (baseline projection)
!
                   CALL FFGCVE ( %VAL(FPTR), %VAL(IND_COL), %VAL(INT8(1)), &
     &                           %VAL(INT8(1)), %VAL(INT8(MAP%NUM_CC)), &
     &                           %VAL(0.0), ARR_R4, ANYF, FT_STATUS )
                   IF ( FT_STATUS .NE. 0 ) THEN
                        CALL FT_PRINTERROR ( 4834, IUER, 'GET_FITS_MAP',  &
     &                                       FT_STATUS )
                        CALL ERR_LOG ( 4835, IUER, 'GET_FITS_MAP', 'Failure to '// &
     &                      'get array of X coordinates of the clean components '// &
     &                      'from the fits file '//FINAM )
                        DEALLOCATE ( KEYS )
                        RETURN 
                   END IF
!
! ---------------- ... and store it
!
                   DO 440 J4=1,MAP%NUM_CC
                      MAP%COOR_CC(1,J4) = ARR_R4(J4)*DEG__TO__RAD
 440               CONTINUE 
                   FL_X = .TRUE.
                 ELSE IF ( KEYS(J3,2)(12:19) == 'DELTAY  ' ) THEN
!
! ---------------- Found the keyword DELTAY. Get the column index
!
                   CALL CHIN ( KEYS(J3,2)(6:8), IND_COL )
                   IF ( IND_COL < 1 .OR. IND_COL > 999 ) THEN
                        CALL ERR_LOG ( 4836, IUER, 'GET_FITS_MAP', 'Failure to '// &
     &                      'decode TYYPE part of the FLUX keyword of the fits '// &
     &                      'file '//FINAM )
                        DEALLOCATE ( KEYS )
                        RETURN 
                   END IF
!
! ---------------- Get array of V-coordinates (baseline projection)
!
                   CALL FFGCVE ( %VAL(FPTR), %VAL(IND_COL), %VAL(INT8(1)), &
     &                           %VAL(INT8(1)), %VAL(INT8(MAP%NUM_CC)), &
     &                           %VAL(0.0), ARR_R4, ANYF, FT_STATUS )
                   IF ( FT_STATUS .NE. 0 ) THEN
                        CALL FT_PRINTERROR ( 4836, IUER, 'GET_FITS_MAP', &
     &                                       FT_STATUS )
                        CALL ERR_LOG ( 4837, IUER, 'GET_FITS_MAP', 'Failure to '// &
     &                      'get array of Y coordinates of the clean components '// &
     &                      'from the fits file '//FINAM )
                        DEALLOCATE ( KEYS )
                        RETURN 
                   END IF
!
! ---------------- ... and store it
!
                   DO 450 J5=1,MAP%NUM_CC
                      MAP%COOR_CC(2,J5) = ARR_R4(J5)*DEG__TO__RAD
                      IF ( IVRB .GE. 2 ) THEN
                           WRITE ( 6, 210 ) J5, MAP%COOR_CC(1,J5)*RAD__TO__MAS, &
     &                                          MAP%COOR_CC(2,J5)*RAD__TO__MAS, &
     &                                          MAP%FLUX_CC(J5)
 210                       FORMAT ( 'Comp: ', I4, ' Coo: ', F8.3, 1X, F8.3, ' Flux: ', F10.6  )
                      END IF
 450               CONTINUE 
                   FL_Y = .TRUE.
              END IF
 430       CONTINUE 
           IF ( FL_FLUX .AND. FL_X .AND. FL_Y ) THEN
                MAP%STATUS_CC = SMP__LOAD
              ELSE 
                WRITE ( 6, * ) 'FL_FLUX = ', FL_FLUX, &
     &                         ' FL_X = ', FL_X, ' FL_Y = ', FL_Y
                CALL ERR_LOG ( 4838, IUER, 'GET_FITS_MAP', 'Failure to find '// &
     &              'arrays with clean components in the fits file '//FINAM )
                DEALLOCATE ( KEYS )
                RETURN 
           END IF
      END IF  ! fl_ccc
!
      IF ( FL_IMA ) THEN
!
! ======== Read the image in the form of a two-dimension pixel array
!
! -------- Allocate dynamic memory
!
           ALLOCATE ( MAP%IMAGE(MAP%DIM1,MAP%DIM2), STAT=IER )
           IF ( IER .NE. 0 ) THEN
                CALL CLRCH ( STR )
                CALL IINCH ( 4*MAP%DIM1*MAP%DIM2, STR )
                CALL ERR_LOG ( 4839, IUER, 'GET_FITS_MAP', 'Failure to '// &
     &              'allocate '//STR(1:I_LEN(STR))//' bytes of dynamic '// &
     &              'memory for the image' )
                DEALLOCATE ( KEYS )
                RETURN 
           END IF
           MAP%STATUS_MAP = SMP__ALLC
!
! -------- Position to the 2nd header unit
!
           CALL FFMAHD ( %VAL(FPTR), %VAL(KH), HDU_TYPE, FT_STATUS )
           IF ( FT_STATUS .NE. 0 ) THEN
                CALL FT_PRINTERROR ( 4840, IUER, 'GET_FITS_MAP', FT_STATUS )
                CALL ERR_LOG ( 4840, IUER, 'GET_FITS_MAP', 'Error in '// &
     &              'an attempt to move to the first header unit of '// &
     &              'the fits file '//FINAM )
                CALL FFITS_CLOSE ( FPTR, -2 )
           END IF
!
! -------- Clean the image
!
           CALL NOUT_R4 ( MAP%DIM1*MAP%DIM2, MAP%IMAGE ) 
!
! -------- Copy the pixel array of the image to MAP%IMAGE
!
           NULLVAL = 0.0
           GROUP = 1
           CALL FFGPVE ( %VAL(FPTR), %VAL(GROUP), %VAL(INT8(1)), &
     &                   %VAL(INT8(MAP%DIM1*MAP%DIM2)), %VAL(NULLVAL), &
     &                   MAP%IMAGE, ANYF, FT_STATUS )
           IF ( FT_STATUS .NE. 0 ) THEN
                CALL FT_PRINTERROR ( 4840, IUER, 'GET_FITS_MAP', FT_STATUS )
                CALL ERR_LOG ( 4841, IUER, 'GET_FITS_MAP', 'Error in '// &
     &              'attempt to get the image from the fits file' )
                CALL FFITS_CLOSE ( FPTR, -2 )
                DEALLOCATE ( KEYS )
                RETURN
           END IF
           MAP%STATUS_MAP = SMP__LOAD
      END IF ! fl_ima
!
      MAP%FINAM = FINAM 
!
! --- Close the file and release the logical unit
!
      CALL FFITS_CLOSE ( FPTR, -2 )
      DEALLOCATE ( KEYS )
!
      CALL ERR_LOG ( 0, IUER )
      RETURN
      END  SUBROUTINE  GET_FITS_MAP  !#!#
