      PROGRAM    BSP_PLOT
! ************************************************************************
! *                                                                      *
! *   Program  BSP_PLOT  analyzes contents of file of site displacements *
! *   in BSPPOS format presented as a sum of an offset, linear drift,    *
! *   and an expansion over B-spline basis. Popular VLBI processing      *
! *   program Solve generates such a file from the raw listing of        *
! *   a solution using utility getpar.                                   *
! *                                                                      *
! *   BSP_PLOT generates time series of site displacements, makes their  *
! *   plots and in a case if the spine had nodes with multiplicity,      *
! *   it computes jump(s) in positions and their uncertainties.          *
! *                                                                      *
! *   Usage:                                                             *
! *                                                                      *
! *      Usage: bsp_plot bsp-file sta_nam [-uen] [-plot] [-table]        *
! *            [-dates date_min date_max step_in_dates]                  *
! *                                                                      *
! *      bsp-file -- name of the file in BSPPOS format that has          *
! *                  estimates of site positions, velocities,            *
! *                  coefficients of B-spline expansion and their        *
! *                  covariance matrix. The file is generated by utility *
! *                  getpar.                                             *
! *                                                                      *
! *      sta-name -- station name.                                       *
! *                                                                      *
! *      -uen     -- by default plots and position jumps are computed    *
! *                  in Cartesian crust-fixed coordinate system.         *
! *                  When this option is set, then position time series  *
! *                  and discontinuities are computed in local           *
! *                  topocentric coordinates system: Up, East, and North.*
! *                                                                      *
! *      -plot    -- this option enables plotting. Only the amount in    *
! *                  position discontinuity(ies) is printed.             *
! *                                                                      *
! *      -table   -- this option enables printing a table with position  *
! *                  time series plotting.                               *
! *                                                                      *
! *      -dates   -- specifies the date range and step. If the argument  *
! *                  is omitted, the begin and start range is taken      *
! *                  from the station position file and the step is set  *
! *                  to 10 days.                                         *
! *      date_min -- the beginning date of the interval. The date is in  *
! *                  YYYY.MM.DD_hh:mm:ss format.                         *
! *      date_min -- the end date of the interval. The date is in        *
! *                  YYYY.MM.DD_hh:mm:ss format.                         *
! *      step_in_days -- time step in days.                              *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ### 08-MAR-2005    BSP_PLOT   v3.1 (d)  L. Petrov  17-SEP-2025 ###  *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      INCLUDE   'astro_constants.i'
      INCLUDE   'vtd.i'
      TYPE     ( BSPPOS__TYPE ) :: BSP(VTD__M_BSP)
      CHARACTER  FILBSP*128, STA_NAM*8, STR*128, DATE_BEG_STR*20, &
     &           DATE_END_STR*20, DATE_STP_STR*20
      LOGICAL*1  FL_PLOT, FL_UEN, FL_TABLE, FL_LOAD_COV
      REAL*8     TAI_BEG, TAI_END, TIM_STP
      INTEGER*4  L_BSP, MJD_BEG, MJD_END, I_STA, KA, J1, J2, J3, IUER
      INTEGER*4, EXTERNAL :: GET_UNIT, ILEN, I_LEN, LINDEX, LTM_DIF
!
      CALL CLRCH ( DATE_BEG_STR )
      CALL CLRCH ( DATE_END_STR )
      CALL CLRCH ( DATE_STP_STR )
      MJD_BEG     = 0
      MJD_END     = 0
      TAI_BEG     = 0.0D0
      TAI_END     = 0.0D0
      TIM_STP     = 10.0D0*86400.0D0
      FL_LOAD_COV = .TRUE.
      FL_PLOT     = .FALSE.
      FL_UEN      = .FALSE.
      FL_TABLE    = .FALSE.
!
      IF ( IARGC() .LT. 2 ) THEN
           WRITE ( 6, '(A)' ) 'Usage: bsp_plot bsp-file sta_nam [-uen] [-plot] [-table] '// &
     &                        '[-dates date_min date_max step_in_days]'
           CALL EXIT ( 0 ) 
         ELSE
           CALL GETARG ( 1, FILBSP )
           CALL GETARG ( 2, STA_NAM )
           KA = 3 
           IF ( IARGC() > 2 ) THEN
                DO 410 J1=3,IARGC()
                   IF ( KA > IARGC() ) GOTO 810
                   CALL GETARG ( KA, STR )
                   IF ( STR == '-uen' ) THEN
                        FL_UEN = .TRUE.
                      ELSE IF ( STR == '-plot' ) THEN
                        FL_PLOT = .TRUE.
                      ELSE IF ( STR == '-table' ) THEN
                        FL_TABLE = .TRUE.
                      ELSE IF ( STR == '-dates' ) THEN
                        IF ( IARGC() < KA + 3 ) THEN
                             WRITE ( 6, * ) 'Too few arguments: three arguments should follow -dates'
                             CALL EXIT ( 1 )
                        END IF
!
                        KA = KA + 1
                        CALL GETARG ( KA, DATE_BEG_STR)
!
                        KA = KA + 1
                        CALL GETARG ( KA, DATE_END_STR )
!
                        KA = KA + 1
                        CALL GETARG ( KA, DATE_STP_STR )
                      ELSE 
                        WRITE ( 6, * ) 'Unsupported option '//STR(1:I_LEN(STR))
                        CALL EXIT ( 1 )
                   END IF
                   KA = KA + 1
 410            CONTINUE
 810            CONTINUE
           END IF
      END IF
!
      IF ( ILEN(DATE_BEG_STR) > 0 ) THEN
!
! -------- Parse dates
!
           IUER = -1
           CALL DATE_TO_TIME ( DATE_BEG_STR, MJD_BEG, TAI_BEG, IUER )
           IF ( IUER .NE. 0 ) THEN
                WRITE ( 6, * ) 'Error in parsing date_beg argument'
                CALL EXIT ( 1 )
           END IF
           IUER = -1
           CALL DATE_TO_TIME ( DATE_END_STR, MJD_END, TAI_END, IUER )
           IF ( IUER .NE. 0 ) THEN
                WRITE ( 6, * ) 'Error in parsing date_end argument'
                CALL EXIT ( 1 )
           END IF
           IF ( INDEX ( DATE_STP_STR, '.' ) < 1 ) THEN
                DATE_STP_STR = TRIM(DATE_STP_STR)//'.0'
           END IF
           READ ( UNIT=DATE_STP_STR, FMT='(F12.5)' ) TIM_STP
           TIM_STP = 86400.0D0*TIM_STP
      END IF
!!   write ( 6, * ) ' BEG: ', MJD_beg, tai_beg, ' END: ', mjd_end, tai_end, ' tim_stp= ', tim_stp ;  call flush ( 6 ) ! %%%%
!
! --- Read and parse the file with estimates of the B-spline model
! --- of station modeion
!
      IUER = -1
      CALL VTD_READ_BSP ( FILBSP, FL_LOAD_COV, VTD__M_BSP, L_BSP, BSP, IUER )
      IF ( IUER .NE. 0 ) THEN
           CALL EXIT ( 1 )
      END IF
!
! --- Search for station name
!
      I_STA = 0
      DO 420 J2=1,L_BSP
         IF ( BSP(J2)%STATION == STA_NAM ) THEN
              I_STA = J2
         END IF
 420  CONTINUE 
      IF ( I_STA == 0 ) THEN
           IUER = -1
           CALL ERR_LOG ( 8001, IUER, 'BSP_PLOT', 'There are no '// &
     &         'parameters of non-linear motion modeled with '// &
     &         'B-spline for station '//STA_NAM(1:I_LEN(STA_NAM))// &
     &         ' in file '//FILBSP )
           CALL EXIT ( 1 )
      END IF
!
      IF ( MJD_BEG == 0 .AND. MJD_END == 0 ) THEN
!
! -------- Well, the user does not specify start/stop dates.
! -------- Then we extract these dates from 
!
           MJD_BEG = (BSP(I_STA)%TIM(1) - 43200.0D0)/86400.0D0 + J2000__MJD
           TAI_BEG = (BSP(I_STA)%TIM(1) - 43200.0D0) - 86400.0D0*(MJD_BEG - J2000__MJD)
           IF ( TAI_BEG < 0.0D0 ) THEN
                MJD_BEG = MJD_BEG - 1
                TAI_BEG = TAI_BEG + 86400.D0
           END IF
           MJD_END = (BSP(I_STA)%TIM(BSP(I_STA)%L_NOD) - 43200.0D0)/86400.0D0 + J2000__MJD
           TAI_END = (BSP(I_STA)%TIM(BSP(I_STA)%L_NOD) - 43200.0D0) - 86400.0D0*(MJD_END - J2000__MJD)
           IF ( TAI_END < 0.0D0 ) THEN
                MJD_END = MJD_END - 1
                TAI_END = TAI_END + 86400.D0
           END IF
      END IF
!
! --- Build time series, make plots and print station position
! --- discontinuity(ies)
!
      IUER = -1
      CALL  BSP_SER ( BSP(I_STA), FL_UEN, FL_PLOT, FL_TABLE, &
     &                MJD_BEG, TAI_BEG, MJD_END, TAI_END, &
     &                TIM_STP, IUER )
      IF ( IUER .NE. 0 ) THEN
           CALL EXIT ( 1 )
      END IF
!
      END  PROGRAM  BSP_PLOT !#!#
!
! ------------------------------------------------------------------------
!
      SUBROUTINE BSP_SER ( BSP, FL_UEN, FL_PLOT, FL_TABLE, MJD_BEG, TAI_BEG, &
     &                MJD_END, TAI_END, TIM_STP, IUER )
! ************************************************************************
! *                                                                      *
! *   Routine BSP_SER computes the time series of station positions      *
! *   represented as a sum of a linear model and the expansion over the  *
! *   B-spline basis, makes plots and prints station position            *
! *   discontinuity(ies).                                                *
! *                                                                      *
! * _________________________ Input parameters: ________________________ *
! *                                                                      *
! * BSP      ( BSPPOS__TYPE ) -- Data structure with information about   *
! *                              station position evolution in a form    *
! *                              of B-spline expansion.                  *
! * FL_UEN   ( LOGICAL*1 ) -- If .TRUE., then the pklot will show        *
! *                           cooddinates in Up, East and North basis.   *
! *                           The output table will not be modified.     *
! * FL_TABLE ( LOGICAL*1 ) -- If .TRUE., then print the table with       *
! *                           station position time series.              *
! * FL_PLOT  ( LOGICAL*1 ) -- If .TRUE., then display plot table with    *
! *                           station position time series.              *
! * MJD_BEG  ( INTEGER*4 ) -- MJD date of the beginning the interval for *
! *                           the time series of station positions in    *
! *                           days.                                      *
! * TAI_BEG  ( REAL*8    ) -- TAI date of the beginning the interval for *
! *                           the time series of station positions in    *
! *                           seconds.                                   *
! * MJD_END  ( INTEGER*4 ) -- MJD date of the end of the interval for    *
! *                           the time series of station positions in    *
! *                           days.                                      *
! * TAI_END  ( INTEGER*4 ) -- TAI date of the end of the interval for    *
! *                           the time series of station positions in    *
! *                           seconds.                                   *
! * TIM_STP  ( REAL*8    ) -- Time step of the output time series        *
! *                           in days.                                   *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ###  26-JUL-2008    BSP_SER   v2.0 (d)  L. Petrov  06-JUL-2025 ###  *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      INCLUDE   'astro_constants.i'
      INCLUDE   'solve.i'
      INCLUDE   'vtd.i'
      INCLUDE   'diagi.i'
      TYPE     ( BSPPOS__TYPE ) :: BSP
      TYPE     ( DIAGI_STRU ) ::  DIAGI_S
      LOGICAL*1  FL_UEN, FL_PLOT, FL_TABLE
      INTEGER*4  MJD_BEG, MJD_END, IUER
      REAL*8     TAI_BEG, TAI_END, TIM_STP
      INTEGER*4  MPN, MP, MS
      PARAMETER  ( MPN = 128, MP = VTD__M_NOD*MPN )
      PARAMETER  ( MS = 64*1024 ) 
      REAL*8     T(MP), TARG(MS), COO(MS,6), ERR(MS,6), XYZ_TO_UEN(3,3), &
     &           COV_XYZ(3,3), COV_UEN(3,3), MAT_33(3,3), &
     &           VEC1(3), VEC2(3), EPS, EPS_ARG, T1(MP), C1(MP,3), E1(MP,3), &
     &           TIM_OBS, TIM(MS), VEL_PRIOR(3), VEL_AFTER(3), TIM_2ND, &
     &           TIM_PRV, RANGE_VEL
      REAL*8,    ALLOCATABLE :: MAT(:,:)
      PARAMETER  ( EPS     = 1.D0 )
      PARAMETER  ( EPS_ARG = 0.001D0 )
      CHARACTER  STR*32, STR1*32, CMP(3)*1, ZAG*128, UNIT*128, OUT(MPN)*256
      INTEGER*4  LP, KP, LN, KNA, KJUMP, NTIM, MJD_OBS, IDAY, NO, J1, J2, &
     &           J3, J4, J5, J6, J7, J8, J9, J10, J11, J12, J13, J14, &
     &           J15, J16, J17, J18, IER
      INTEGER*4  IBST, ILST, IOST, IPST, IWST, IDEV, ICL1, ICL2, ICL3
      INTEGER*4  DIAGI_LEN
      REAL*8,    EXTERNAL :: EBSPL_VAL_R8, LIN_BSPL_SIG_R8, LIN_BSPL_COV_R8 
      CHARACTER, EXTERNAL :: MJDSEC_TO_DATE*30, TIM_TO_DATE*23, GET_CDATE*19
!
      RANGE_VEL = 1.0D0 * YEAR__TO__SEC
!
      LP = MPN*(BSP%L_NOD-1)
      NO = 0
      KP = 0
      LN = 0
      E1 = 1.0D-5
!
! --- Generate arrays of stations positions and its uncertainties 
! --- at segments between nodes. 
! --- MPN -- the number of points in a single segment
!
      DO 410 J1=1,BSP%L_NOD-1
         IF ( (BSP%TIM(J1+1) - BSP%TIM(J1)) < EPS ) GOTO 410
         DO 420 J2=1,MPN
            KP = KP + 1
!
! --------- Time argument for the J1-th segment
!
            T(KP) = BSP%TIM(J1) + (J2-1)*(BSP%TIM(J1+1) - BSP%TIM(J1))/(MPN-1)
            TARG(KP) = 2000.0 + T(KP)/(86400.0D0*365.25D0)
            IF ( J2 ==   1 ) THEN 
                 T(KP) = T(KP) + EPS
                 TARG(KP) = TARG(KP) + EPS_ARG
              ELSE IF ( J2 == MPN ) THEN
                 T(KP) = T(KP) - EPS
                 TARG(KP) = TARG(KP) - EPS_ARG
            END IF
!
! --------- Get position and errors for the given epoch
!
            DO 430 J3=1,3
               COO(KP,J3) = EBSPL_VAL_R8 ( BSP%L_NOD, BSP%L_DEG, T(KP), &
     &                                     BSP%TIM(1), BSP%SPL(1-BSP%L_DEG,J3) )* &
     &                      1.0D3
               ERR(KP,J3) = LIN_BSPL_SIG_R8 ( BSP%L_NOD, BSP%L_DEG, T(KP), &
     &                                        BSP%TIM_REF, BSP%TIM(1), &
     &                                        BSP%COV(1-BSP%L_DEG,1-BSP%L_DEG,J3,J3) )*1.0D3
               VEC1(J3) = COO(KP,J3)
 430        CONTINUE 
            IF ( FL_UEN ) THEN
!
! -------------- Convert XYZ i UEN coordinates
!
                 IF ( KP == 1 ) THEN
                      CALL MAKE_XYZ_TO_UEN  ( BSP%COO, XYZ_TO_UEN )
                 END IF
                 DO 440 J4=1,3
                    DO 450 J5=1,3
                       COV_XYZ(J4,J5) = LIN_BSPL_COV_R8 ( BSP%L_NOD, BSP%L_DEG, &
     &                                      T(KP), T(KP), BSP%TIM_REF, &
     &                                      BSP%TIM(1), &
     &                                      BSP%COV(1-BSP%L_DEG,1-BSP%L_DEG,J4,J5) ) 
 450                CONTINUE 
 440             CONTINUE 
                 CALL MUL_MM_II_I ( 3, 3, XYZ_TO_UEN, 3, 3, COV_XYZ, &
     &                              3, 3, MAT_33, -2 )
                 CALL MUL_MM_IT_I ( 3, 3, MAT_33, 3, 3, XYZ_TO_UEN, &
     &                              3, 3, COV_UEN, -2 )
                 CALL MUL_MV_IV_V ( 3, 3, XYZ_TO_UEN, 3, VEC1, 3, VEC2, -2 )
                 DO 460 J6=1,3
                    ERR(KP,J6) = DSQRT ( COV_UEN(J6,J6) )*1.D3
                    COO(KP,J6) = VEC2(J6)
 460             CONTINUE 
            END IF
            IF ( J2 == 1 .OR. J2 == MPN ) THEN
                 LN = LN + 1
                 T1(LN) = TARG(KP) 
                 C1(LN,1) = COO(KP,1)
                 C1(LN,2) = COO(KP,2)
                 C1(LN,3) = COO(KP,3)
            END IF
 420     CONTINUE 
 410  CONTINUE 
!
      IF ( FL_UEN ) THEN
           CMP(1) = 'U'
           CMP(2) = 'E'
           CMP(3) = 'N'
         ELSE 
           CMP(1) = 'X'
           CMP(2) = 'Y'
           CMP(3) = 'Z'
      END IF
!
! --- Compute estimates of station discontinuties and their uncertainties
!
      KJUMP = 0
      DO 470 J7=2,BSP%L_NOD-1  ! Cycle over disconnuties
         KNA = 0
!
! ------ Find the node with discontinuity
!
         DO 480 J8=J7+1,BSP%L_NOD
            IF ( BSP%TIM(J8) - BSP%TIM(J7) < EPS ) THEN
                 KNA = J8
            END IF
 480     CONTINUE 
         IF ( BSP%TIM(J7) - BSP%TIM(J7-1) < EPS ) THEN
              KNA = 0
         END IF
         IF ( BSP%L_DEG == 0 ) KNA = J7+1
!
         IF ( KNA > 0 ) THEN
              DO 490 J9=1,3
!
! -------------- Compute the position just before the discontinuity
!
                 VEC1(J9) = EBSPL_VAL_R8 ( BSP%L_NOD, BSP%L_DEG, BSP%TIM(J7)+EPS, &
     &                                     BSP%TIM(1), BSP%SPL(1-BSP%L_DEG,J9) )
!
! -------------- Compute the position just after the discontinuity and 
! -------------- and store the difference in VEC1
!
                 VEC1(J9) = VEC1(J9) - &
     &                      EBSPL_VAL_R8 ( BSP%L_NOD, BSP%L_DEG, BSP%TIM(J7)-EPS, &
     &                                     BSP%TIM(1), BSP%SPL(1-BSP%L_DEG,J9) )
 490          CONTINUE 
!
              DO 4100 J10=1,3
                 DO 4110 J11=1,3
!
! ----------------- Compute the covariance matrix of the discontinuity
!
                    COV_XYZ(J10,J11) = LIN_BSPL_COV_R8 ( BSP%L_NOD, BSP%L_DEG, &
     &                                     BSP%TIM(J7)-EPS, BSP%TIM(J7)+EPS, &
     &                                     BSP%TIM_REF, BSP%TIM(1), &
     &                                     BSP%COV(1-BSP%L_DEG,1-BSP%L_DEG,J10,J11) ) 
 4110            CONTINUE 
 4100         CONTINUE 
!
! ----------- Get the jump epoch
!
              KJUMP = KJUMP + 1
              IER = -1
              STR = MJDSEC_TO_DATE ( J2000__MJD, BSP%TIM(J7) + 43200.0D0, IER )
              NO = NO + 1
              WRITE ( OUT(NO), 110 ) BSP%STATION, KJUMP, BSP%STATION, STR(1:16)
 110          FORMAT ( '# Dscnt: ',A, ' Discontinuity #', I1, ' for station ', A, &
     &                 ' at epoch ', A )
              IF ( FL_UEN ) THEN
!
! ---------------- Transform the discontinuity to Up-East-North 
! ---------------- topocentric coordinate system
!
                   CALL MAKE_XYZ_TO_UEN  ( BSP%COO, XYZ_TO_UEN )
                   CALL MUL_MM_II_I ( 3, 3, XYZ_TO_UEN, 3, 3, COV_XYZ, &
     &                                3, 3, MAT_33, -2 )
                   CALL MUL_MM_IT_I ( 3, 3, MAT_33, 3, 3, XYZ_TO_UEN, &
     &                                3, 3, COV_UEN, -2 )
                   CALL MUL_MV_IV_V ( 3, 3, XYZ_TO_UEN, 3, VEC1, 3, VEC2, -2 )
                   DO 4120 J12=1,3
                      IF ( COV_UEN(J12,J12) < 0.0 ) THEN
                           WRITE ( 6, * ) 'WARNING: COVARIANCE IS WRONG!!!'
                           COV_UEN(J12,J12) = - COV_UEN(J12,J12) 
                      END IF
                      NO = NO + 1
                      WRITE ( OUT(NO), 120 ) BSP%STATION, CMP(J12), VEC2(J12)*1.D3, &
     &                                 DSQRT ( COV_UEN(J12,J12) )*1.D3
  120                 FORMAT ( '# Pos:   ', A, 1X, A, '-coord   ', &
     &                         F8.1, ' -+ ', F4.1, ' mm ' )
 4120              CONTINUE 
                 ELSE 
!
! ---------------- Write down the station positoins
!
                   DO 4130 J13=1,3
                      NO = NO + 1
                      WRITE ( OUT(NO), 120 ) BSP%STATION, CMP(J13), VEC1(J13)*1.D3, &
     &                                       DSQRT ( MAX ( COV_XYZ(J13,J13), 1.D-10 ) )*1.D3
 4130              CONTINUE 
              END IF
         END IF
 470  CONTINUE 
      IF ( FL_TABLE ) THEN
           NTIM = ((MJD_END*86400.0D0 + TAI_END) - (MJD_BEG*86400.0D0 + TAI_BEG))/TIM_STP + 1
!
! -------- Compute the end epoch for the range close to the beginning
! -------- the interval used for position estimates. This epoch range 
! -------- is RANGE_VEL or shorter if the second epoch of the B-spline
! -------- is close to the to the beginning the interval than RANGE_VEL
!
           TIM_2ND = MIN ( BSP%TIM(2), BSP%TIM(1) + RANGE_VEL )
!
! -------- Compute the start epoch for the range close to the end 
! -------- the interval used for position estimates
!
           TIM_PRV = MAX ( BSP%TIM(BSP%L_NOD-1), BSP%TIM(BSP%L_NOD) - RANGE_VEL )
!
           DO 4140 J14=1,3
!
! ------------ Compute the representative velocity used for extrapolation 
! ------------ earlier than the first epoch used for parameter estimation
!
              VEL_PRIOR(J14) = ( EBSPL_VAL_R8 ( BSP%L_NOD, BSP%L_DEG, TIM_2ND - EPS, &
     &                                          BSP%TIM(1), BSP%SPL(1-BSP%L_DEG,J14) ) - &
     &                           EBSPL_VAL_R8 ( BSP%L_NOD, BSP%L_DEG, BSP%TIM(1) + EPS, &
     &                                          BSP%TIM(1), BSP%SPL(1-BSP%L_DEG,J14) )   &
     &                         ) / ( TIM_2ND - BSP%TIM(1) )
!
! ------------ Compute the representative velocity used for extrapolation 
! ------------ beyond the last epoch used for parameter estimation
!
              VEL_AFTER(J14) = ( EBSPL_VAL_R8 ( BSP%L_NOD, BSP%L_DEG, BSP%TIM(BSP%L_NOD) - EPS, &
     &                                          BSP%TIM(1), BSP%SPL(1-BSP%L_DEG,J14) ) - &
     &                           EBSPL_VAL_R8 ( BSP%L_NOD, BSP%L_DEG, TIM_PRV, &
     &                                          BSP%TIM(1), BSP%SPL(1-BSP%L_DEG,J14) )   &
     &                         ) / ( BSP%TIM(BSP%L_NOD) - TIM_PRV )
 4140      CONTINUE 
!
! -------- Prepare the table header
!
           IER = -1
           STR  = TIM_TO_DATE ( BSP%TIM(1)         + 43200.0D0 + EPS, IER )
           IER = -1
           STR1 = TIM_TO_DATE ( BSP%TIM(BSP%L_NOD) + 43200.0D0 + EPS, IER )
!
           WRITE ( 6, '(A)' ) STA_POS_EVO__LABEL
           WRITE ( 6, '(A)' ) '# '
           WRITE ( 6, '(A)' ) '# Generated on '//GET_CDATE()
           WRITE ( 6, '(A)' ) '# Input file: '//BSP%FILE_NAME
           WRITE ( 6, '(A)' ) '# '
           IF ( NO > 0 ) THEN
!
! ------------- Print information about position discontinuities
!
                DO 4150 J15=1,NO
                   WRITE ( 6, '(A)' ) TRIM(OUT(J15))
 4150           CONTINUE 
           END IF
           WRITE ( 6, 210 ) BSP%STATION, STR(1:10),  (BSP%VEL + VEL_PRIOR)*YEAR__TO__SEC*1000.0D0, &
     &                      BSP%STATION, BSP%VEL*YEAR__TO__SEC*1000.0D0, &
     &                      BSP%STATION, STR1(1:10), (BSP%VEL + VEL_AFTER)*YEAR__TO__SEC*1000.0D0
 210       FORMAT ( '# Vel:   ', A, ' Vel prior to ', A, 2X, 3( F6.2, 1X), ' mm/year' / &
                    '# Vel:   ', A, ' Global vel             ', 2X, 3( F6.2, 1X), ' mm/year' / &
     &              '# Vel:   ', A, ' Vel after to ', A, 2X, 3( F6.2, 1X), ' mm/year' )
           WRITE ( 6, '(A)' ) '#'
           WRITE ( 6, '(A)' ) '# ================================================================================'
           WRITE ( 6, '(A)' ) '# Byte-by-byte Description:'
           WRITE ( 6, '(A)' ) '# --------------------------------------------------------------------------------'
           WRITE ( 6, '(A)' ) '#   Bytes Format Units Label   Explanations'
           WRITE ( 6, '(A)' ) '# --------------------------------------------------------------------------------'
           WRITE ( 6, '(A)' ) '#'
           WRITE ( 6, '(A)' ) '#    1- 8 A8    ---    Name    IVS station name'
           WRITE ( 6, '(A)' ) '#   10-19 A10   ---    CalDate Calendar date of the epoch in YYYY.MM.DD format'
           WRITE ( 6, '(A)' ) '#   21-25 I5    days   MJDDate MJD of the epoch'
           WRITE ( 6, '(A)' ) '#   27-39 F13.4 meter  Xpos    X-coordine of station position on a given epoch'
           WRITE ( 6, '(A)' ) '#   41-53 F13.4 meter  Xpos    Y-coordine of station position on a given epoch'
           WRITE ( 6, '(A)' ) '#   55-67 F13.4 meter  Xpos    Z-coordine of station position on a given epoch'
           WRITE ( 6, '(A)' ) '# --------------------------------------------------------------------------------'
           WRITE ( 6, '(A)' ) '#'
           WRITE ( 6, '(A)' ) '# Name   Date       MJD    X-position    Y-position    Z-position'
           WRITE ( 6, '(A)' ) '#'
!
! -------- Print table
!
           DO 4160 J16=1,NTIM
!
! ----------- Get time epoch that will be put in the table
!
              TIM_OBS = TAI_BEG + (J16-1)*TIM_STP
              IDAY = TIM_OBS/86400.0D0
              MJD_OBS = MJD_BEG + IDAY
              TIM_OBS = TIM_OBS - IDAY*86400.0D0
              TIM(J16) = (MJD_OBS - J2000__MJD)*86400.0D0 + TIM_OBS
              DO 4170 J17=1,3
                 IF ( TIM(J16) - 43200.0D0 < BSP%TIM(1) + EPS ) THEN
!
! ------------------- Extrapolation prior the first epoch of the B-spline
! ------------------- model
!
                      COO(J16,J17) = BSP%COO(J17) + &
     &                               EBSPL_VAL_R8 ( BSP%L_NOD, BSP%L_DEG, BSP%TIM(1) + EPS, &
     &                                              BSP%TIM(1), BSP%SPL(1-BSP%L_DEG,J17) ) + &
     &                               (TIM(J16) - BSP%TIM_REF)*BSP%VEL(J17) + &
     &                               (TIM(J16) - BSP%TIM(1))*VEL_PRIOR(J17)
                    ELSE IF ( TIM(J16) - 43200.0D0 > BSP%TIM(BSP%L_NOD) - EPS ) THEN
!
! ------------------- Extrapolation beyond the last epoch of the B-spline
! ------------------- model
!
                      COO(J16,J17) = BSP%COO(J17) + &
     &                               EBSPL_VAL_R8 ( BSP%L_NOD, BSP%L_DEG, BSP%TIM(BSP%L_NOD) - EPS, &
     &                                              BSP%TIM(1), BSP%SPL(1-BSP%L_DEG,J17) ) + &
     &                               (TIM(J16) - BSP%TIM_REF)*BSP%VEL(J17)  + &
     &                               (TIM(J16) - BSP%TIM(BSP%L_NOD))*VEL_AFTER(J17)
                    ELSE
!
! ------------------- Interpolation within the range of the B-spline model
!
                      COO(J16,J17) = BSP%COO(J17) + &
     &                               EBSPL_VAL_R8 ( BSP%L_NOD, BSP%L_DEG, TIM(J16) - 43200.0D0 , &
     &                                              BSP%TIM(1), BSP%SPL(1-BSP%L_DEG,J17) ) + &
     &                               (TIM(J16) - BSP%TIM_REF)*BSP%VEL(J17)
                 END IF
                 ERR(J16,J17) = 0.001D0
 4170        CONTINUE 
             IER = -1
             STR = MJDSEC_TO_DATE ( MJD_OBS, TIM_OBS, IER )
!
! ---------- Prionte the lint
!
             WRITE ( 6, 130 ) BSP%STATION, STR(1:10), MJD_OBS, COO(J16,1:3)
 130         FORMAT ( A8,1X,A10, 1X, I5, 1X, 4(F13.4,1X) )
!
! ---------- Update array COO in a form suitable for plitting: subtract 
! ---------- position at the reference epoch
!
             COO(J16,1:3) = COO(J16,1:3) - BSP%COO(1:3)
 4160     CONTINUE 
          KP = NTIM
          TARG = TIM/(86400.0D0*365.25D0) + 2000.0D0
      END IF
      IF ( FL_PLOT ) THEN
!
! -------- Prepare plot
!
           CALL DIAGI_DEF ( IBST, ILST, IOST, IPST, IWST, IDEV, ZAG, &
     &                      UNIT, ICL1, ICL2, ICL3, IER )
           DO 4180 J18=1,3
              CALL DIAGI_SETDEF ( IER, 'DIAGI_CTIT', 'Station '//BSP%STATION// &
     &                            '  '//CMP(J18)//'-coordinate (mm)' )
              CALL DIAGI_SETDEF ( IER, 'DIAGI_UNIT', 'Time (years)' )
              CALL DIAGI_SETDEF ( IER, 'DIAGI_ILST', 2 )
              CALL DIAGI_SETDEF ( IER, 'DIAGI_IBST', 4 )
              CALL DIAGI_SETDEF ( IER, 'DIAGI_IWST', 3 )
              CALL DIAGI_SETDEF ( IER, 'DIAGI_IPST', 1 )
              IER = -1
!
! ----------- Clear DIAGI_S object
!
              DIAGI_LEN = LOC(DIAGI_S%STATUS) - LOC(DIAGI_S%IFIRST_FIELD) + 4
              CALL NOUT ( DIAGI_LEN, DIAGI_S )
!
! ----------- Setting up the values of the DIAGI internal data structure for the further
! ----------- plotting
!
              DIAGI_S%IDEV      = IDEV
              DIAGI_S%NCLR      = 1
              DIAGI_S%NPOI(1)   = KP
              DIAGI_S%ADR_X8(1) = LOC(TARG)
              DIAGI_S%ADR_Y8(1) = LOC(COO(1,J18))
              DIAGI_S%ADR_E8(1) = LOC(ERR(1,J18))
              DIAGI_S%LER(1)    = .TRUE.
              DIAGI_S%ICOL(1)   = 1
              DIAGI_S%IBST(1)   = 4
              DIAGI_S%ILST(1)   = 2
              DIAGI_S%IOST(1)   = 1
              DIAGI_S%IPST(1)   = 1
              DIAGI_S%IWST(1)   = 2
!
              DIAGI_S%NPOI(2)   = LN
              DIAGI_S%ADR_X8(2) = LOC(T1)
              DIAGI_S%ADR_Y8(2) = LOC(C1(1,J18))
              DIAGI_S%ADR_E8(2) = LOC(E1(1,J18))
              DIAGI_S%LER(2)    = .TRUE.
              DIAGI_S%ICOL(2)   = 1
              DIAGI_S%IBST(2)   = 1
              DIAGI_S%ILST(2)   = 1
              DIAGI_S%IOST(2)   = 1
              DIAGI_S%IPST(2)   = 3
              DIAGI_S%IWST(2)   = 1
!
              DIAGI_S%ICLR      = 2
              DIAGI_S%XMIN      = 1.0
              DIAGI_S%XMAX      = 0.0
              DIAGI_S%YMIN      = 1.0
              DIAGI_S%YMAX      = 0.0
              DIAGI_S%ZAG       = 'Station '//BSP%STATION// &
     &                            '  '//CMP(J18)//'-coordinate (mm)' 
              DIAGI_S%NAME      = BSP%STATION//'_'//CMP(J18)//'-coor'
              DIAGI_S%ARG_UNITS = 'Time (years)' 
              DIAGI_S%ITRM      = 0
              DIAGI_S%IBATCH    = 0
              DIAGI_S%STATUS    = DIA__DEF
!
! ----------- Calling the main routine of DiaGI
!
              IER = -1
              CALL DIAGI     ( DIAGI_S, IER )
 4180       CONTINUE 
      END IF
!
      CALL ERR_LOG ( 0, IUER )
      RETURN
      END  SUBROUTINE  BSP_SER  !#!  
!
! ------------------------------------------------------------------------
!
      FUNCTION EBSPL_SIG_R8 ( MN, DEG, ARG, ARG_VEC, BCOE_VEC, BCOE_COV )
! ************************************************************************
! *                                                                      *
! *   Routine  EBSPL_VAL_R8  computes the uncertainty of a function      *
! *   represented as an expansion over the B-spline basis at point with  *
! *   argument ARG using coefficients of its expansion.
! *                                                                      *
! * _________________________ Input parameters: ________________________ *
! *                                                                      *
! *       MN ( INTEGER*4 ) -- The number of knots of the B-spline.       *
! *      DEG ( INTEGER*4 ) -- Degree of B-spline.                        *
! *      ARG ( REAL*8    ) -- Argument for which the function is         *
! *                           computed.
! *  ARG_VEC ( REAL*8    ) -- Array of arguments for B-spline at knots.  *
! *                           Dimension: [1:MN].                         *
! * BCOE_VEC ( REAL*8    ) -- Array of B-spline coefficients.            *
! *                           Dimension: [1-DEG:MN-1].                   *
! * BCOE_COV ( REAL*8    ) -- Elemets of the covariance matrix of 
! *                           B-spline coefficient estimates.            *
! *                           Dimension: [1-DEG:MN-1]
! *                                                                      *
! * _________________________ Output parameters: _______________________ *
! *                                                                      *
! * <EBSPL_VAL_R8> ( REAL*8    ) -- value of the function computed using *
! *                                 its expansion over B-spline basis.   *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ### 25-MAR-2010  EBSPL_VAL_R8  v1.0 (d)  L. Petrov 25-MAR-2010 ###  *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      REAL*8     EBSPL_SIG_R8 
      INTEGER*4  MN, DEG
      REAL*8     ARG, ARG_VEC(MN), BCOE_VEC(1-DEG:MN-1), &
     &           BCOE_COV(1-DEG:MN-1,1-DEG:MN-1)
      INTEGER*4  J1, J2, IP
      REAL*8,    EXTERNAL :: BSPL_VAL 
      INTEGER*4, EXTERNAL :: IXMN8 
!
      EBSPL_SIG_R8 = 0.0D0
      IF ( ARG < ARG_VEC(1)  .OR.  ARG > ARG_VEC(MN) ) THEN
!
! -------- If the argument is away from the the interpolation range,
! -------- nothing to do: its value is zero.
!
           RETURN 
      END IF
      IP = IXMN8 ( MN, ARG_VEC, ARG )
      IF ( IP < 1    ) IP = 1 
      IF ( IP > MN-1 ) IP = MN-1
      DO 410 J1=0,DEG
         DO 420 J2=0,DEG
            EBSPL_SIG_R8 = EBSPL_SIG_R8 + &
     &                     BCOE_VEC(IP-J1) * BSPL_VAL ( MN, ARG_VEC, DEG, IP-J1, ARG )* &
     &                     BCOE_VEC(IP-J2) * BSPL_VAL ( MN, ARG_VEC, DEG, IP-J2, ARG )* &
     &                     BCOE_COV(IP-1,IP-2)
 420     CONTINUE 
 410  CONTINUE 
      IF ( EBSPL_SIG_R8 >  0.0 ) THEN
           EBSPL_SIG_R8 = DSQRT ( EBSPL_SIG_R8 )
         ELSE 
           EBSPL_SIG_R8 = 0.0D0
      END IF
      RETURN
      END  FUNCTION EBSPL_SIG_R8  !#!#
!
! ------------------------------------------------------------------------
!
      FUNCTION LIN_BSPL_SIG_R8 ( MN, DEG, ARG, TIM_REF, ARG_VEC, BCOE_COV )
! ************************************************************************
! *                                                                      *
! *   Routine  EBSPL_VAL_R8  computes the uncertainty of a function      *
! *   represented as an expansion over the B-spline basis at point with  *
! *   argument ARG using coefficients of its expansion.
! *                                                                      *
! * _________________________ Input parameters: ________________________ *
! *                                                                      *
! *       MN ( INTEGER*4 ) -- The number of knots of the B-spline.       *
! *      DEG ( INTEGER*4 ) -- Degree of B-spline.                        *
! *      ARG ( REAL*8    ) -- Argument for which the function is         *
! *                           computed.
! *  ARG_VEC ( REAL*8    ) -- Array of arguments for B-spline at knots.  *
! *                           Dimension: [1:MN].                         *
! * BCOE_VEC ( REAL*8    ) -- Array of B-spline coefficients.            *
! *                           Dimension: [1-DEG:MN-1].                   *
! * BCOE_COV ( REAL*8    ) -- Elemets of the covariance matrix of        *
! *                           B-spline coefficient estimates.            *
! *                           Dimension: [1-DEG:MN+1,1-DEG:MN+1]         *
! *                                                                      *
! * _________________________ Output parameters: _______________________ *
! *                                                                      *
! * <EBSPL_VAL_R8> ( REAL*8    ) -- value of the function computed using *
! *                                 its expansion over B-spline basis.   *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! * ### 25-MAR-2010  LIN_BSPL_SIG_R8  v1.1 (d) L. Petrov 16-NOV-2011 ### *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      REAL*8     LIN_BSPL_SIG_R8
      INTEGER*4  MN, DEG
      REAL*8     ARG, TIM_REF, ARG_VEC(MN), BCOE_COV(1-DEG:MN+1,1-DEG:MN+1)
      REAL*8     VEL_SCL, MAT_SCL
      REAL*8,    ALLOCATABLE :: COV(:,:), VEC(:)
      INTEGER*4  J1, J2, J3, J4, IND1, IND2, IP
      REAL*8,    EXTERNAL :: BSPL_VAL 
      INTEGER*4, EXTERNAL :: IXMN8 
!
      LIN_BSPL_SIG_R8 = 0.0D0
      IF ( ARG < ARG_VEC(1)  .OR.  ARG > ARG_VEC(MN) ) THEN
!
! -------- If the argument is away from the the interpolation range,
! -------- nothing to do: its value is zero.
!
           RETURN 
      END IF
      IP = IXMN8 ( MN, ARG_VEC, ARG )
      IF ( IP < 1    ) IP = 1 
      IF ( IP > MN-1 ) IP = MN-1
      VEL_SCL = 1.0D0/(ARG_VEC(MN) - ARG_VEC(1))
      ALLOCATE ( COV(0:DEG+2,0:DEG+2) )
      ALLOCATE ( VEC(0:DEG+2) )
      DO 410 J1=0,DEG+2
         IF ( J1 .LE. DEG ) THEN
              VEC(J1) = BSPL_VAL ( MN, ARG_VEC, DEG, IP-J1, ARG )
              IND1 = IP-J1
              MAT_SCL = 1.0D0
            ELSE IF ( J1 == DEG+1 ) THEN
              IND1 = MN
              VEC(J1) = 1.0D0
              MAT_SCL = 1.0D0
            ELSE IF ( J1 == DEG+2 ) THEN
              IND1 = MN+1
              VEC(J1) = (ARG - TIM_REF)*VEL_SCL
              MAT_SCL = VEL_SCL
         END IF
         DO 420 J2=0,DEG+2
            IF ( J2 .LE. DEG ) THEN
                 IND2 = IP - J2
              ELSE IF ( J2 == DEG+1 ) THEN
                 IND2 = MN
              ELSE IF ( J2 == DEG+2 ) THEN
                 IND2 = MN+1
                 MAT_SCL = MAT_SCL * VEL_SCL
            END IF
            COV(J1,J2) = BCOE_COV(IND1,IND2) * MAT_SCL
 420     CONTINUE 
 410  CONTINUE 
      LIN_BSPL_SIG_R8  = 0.0D0
      DO 430 J3=0,DEG+2
         DO 440 J4=0,DEG+2
            LIN_BSPL_SIG_R8 = LIN_BSPL_SIG_R8 + COV(J3,J4)*VEC(J3)*VEC(J4)
 440     CONTINUE 
 430  CONTINUE 
      DEALLOCATE ( COV )
      DEALLOCATE ( VEC )
!
      IF ( LIN_BSPL_SIG_R8  > 0.0D0 ) THEN
           LIN_BSPL_SIG_R8 = DSQRT ( LIN_BSPL_SIG_R8 )
         ELSE 
           LIN_BSPL_SIG_R8 = 0.0D0
      END IF
!      
      RETURN
      END  FUNCTION  LIN_BSPL_SIG_R8  !#!#
!
! ------------------------------------------------------------------------
!
      FUNCTION LIN_BSPL_COV_R8 ( MN, DEG, ARG1, ARG2, TIM_REF, ARG_VEC, &
     &                           BCOE_COV )
! ************************************************************************
! *                                                                      *
! *   Routine  LIN_BSPL_COV_R8 computes the covariance of a function     *
! *   represented as a sumn of free term, linear drift and the expansion *
! *   over B-spline basis over points with arguments ARG1 and ARG2       *
! *   using coefficients the covariance matrix of the function           *
! *   expansion.                                                         *
! *                                                                      *
! * _________________________ Input parameters: ________________________ *
! *                                                                      *
! *       MN ( INTEGER*4 ) -- The number of knots of the B-spline.       *
! *      DEG ( INTEGER*4 ) -- Degree of B-spline.                        *
! *     ARG1 ( REAL*8    ) -- Argument for which the function is         *
! *                           computed.
! *  ARG_VEC ( REAL*8    ) -- Array of arguments for B-spline at knots.  *
! *                           Dimension: [1:MN].                         *
! * BCOE_COV ( REAL*8    ) -- Elemets of the covariance matrix of        *
! *                           B-spline coefficient estimates.            *
! *                           Dimension: [1-DEG:MN+1,1-DEG:MN+1]         *
! *                                                                      *
! *                           Order of the coeeficients:                 *
! *                           1-DEG:MN-1 -- B-spline coefficients        *
! *                           MN         -- free term                    *
! *                           MN+1       -- linear drift term            *
! *                                                                      *
! *                                                                      *
! * _________________________ Output parameters: _______________________ *
! *                                                                      *
! * <EBSPL_VAL_R8> ( REAL*8    ) -- value of the function computed using *
! *                                 its expansion over B-spline basis.   *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! * ### 16-NOV-2011 LIN_BSPL_COV_R8 v1.0 (d)  L. Petrov 16-NOV-2011 ###  *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      REAL*8     LIN_BSPL_COV_R8
      INTEGER*4  MN, DEG
      REAL*8     ARG1, ARG2, TIM_REF, ARG_VEC(MN), &
     &           BCOE_COV(1-DEG:MN+1,1-DEG:MN+1)
      REAL*8     VEL_SCL, MAT_SCL
      REAL*8,    ALLOCATABLE :: COV(:,:), VEC1(:), VEC2(:)
      INTEGER*4  J1, J2, J3, J4, IND1, IND2, IP1, IP2
      REAL*8,    EXTERNAL :: BSPL_VAL 
      INTEGER*4, EXTERNAL :: IXMN8 
!
      LIN_BSPL_COV_R8 = 0.0D0
      IF ( ARG1 < ARG_VEC(1)  .OR.  ARG1 > ARG_VEC(MN)  .OR.  &
     &     ARG1 < ARG_VEC(1)  .OR.  ARG1 > ARG_VEC(MN)        ) THEN
!
! -------- If the argument is away from the the interpolation range,
! -------- nothing to do: its value is zero.
!
           RETURN 
      END IF
!
      IP1 = IXMN8 ( MN, ARG_VEC, ARG1 )
      IF ( IP1 < 1    ) IP1 = 1 
      IF ( IP1 > MN-1 ) IP1 = MN-1
!
      IP2 = IXMN8 ( MN, ARG_VEC, ARG2 )
      IF ( IP2 < 1    ) IP2 = 1 
      IF ( IP2 > MN-1 ) IP2 = MN-1
!
      VEL_SCL = 1.0D0/(ARG_VEC(MN) - ARG_VEC(1))
!
      ALLOCATE ( COV(0:DEG+2,0:DEG+2) )
      ALLOCATE ( VEC1(0:DEG+2) )
      ALLOCATE ( VEC2(0:DEG+2) )
!
      DO 410 J1=0,DEG+2
         IF ( J1 .LE. DEG ) THEN
              VEC1(J1) = BSPL_VAL ( MN, ARG_VEC, DEG, IP1-J1, ARG1 )
              IND1 = IP1 - J1
              MAT_SCL = 1.0D0
            ELSE IF ( J1 == DEG+1 ) THEN
              IND1 = MN
              VEC1(J1) = 1.0D0
              MAT_SCL = 1.0D0
            ELSE IF ( J1 == DEG+2 ) THEN
              IND1 = MN+1
              VEC1(J1) = (ARG1 - TIM_REF)*VEL_SCL
              MAT_SCL = VEL_SCL
         END IF
         DO 420 J2=0,DEG+2
            IF ( J2 .LE. DEG ) THEN
                 VEC2(J2) = BSPL_VAL ( MN, ARG_VEC, DEG, IP2-J2, ARG2 )
                 IND2 = IP2 - J2
              ELSE IF ( J2 == DEG+1 ) THEN
                  VEC2(J2) = 1.0D0
                 IND2 = MN
              ELSE IF ( J2 == DEG+2 ) THEN
                 IND2 = MN+1
                 VEC2(J2) = (ARG2 - TIM_REF)*VEL_SCL
                 MAT_SCL = MAT_SCL * VEL_SCL
            END IF
            COV(J1,J2) = BCOE_COV(IND1,IND2) * MAT_SCL
 420     CONTINUE 
 410  CONTINUE 
      LIN_BSPL_COV_R8  = 0.0D0
      DO 430 J3=0,DEG+2
         DO 440 J4=0,DEG+2
            LIN_BSPL_COV_R8 = LIN_BSPL_COV_R8 + COV(J3,J4)*VEC1(J3)*VEC2(J4)
 440     CONTINUE 
 430  CONTINUE 
      DEALLOCATE ( COV  )
      DEALLOCATE ( VEC1 )
      DEALLOCATE ( VEC2 )
!      
      RETURN
      END  FUNCTION  LIN_BSPL_COV_R8 !#!#
