      FUNCTION PIMA_AMP_TO_SIGPHS ( SNR, AMP )
! ************************************************************************
! *                                                                      *
! *   Routine  PIMA_AMP_TO_SIGPHS  computes the matematical expectation  *
! *   of variance of phase of a sum of a complex number with both image  *
! *   and real part that are independent and normally distributed with   *
! *   a signal with a given SNR on the basis of the amplitude of the     *
! *   sum. The SNR is determined as a ratio of the amplitude of the      *
! *   signal to the the variance of the image or real path of the        *
! *   complex number with noise.                                         *
! *                                                                      *
! * _________________________ Input parameters: ________________________ *
! *                                                                      *
! * SNR ( REAL*8    ) -- The SNR: the ratio of the sinal to the variance *
! *                      of image or real part of the noise in the form  *
! *                      of a complex number.                            *
! * AMP ( REAL*8    ) -- Amplitude of the sum of a signal and noise.     *
! *                                                                      *
! * _________________________ Output parameters: _______________________ *
! *                                                                      *
! * PIMA_AMP_TO_SIGPHS ( REAL*8    ) -- Mathematical expectation of the  *
! *                                     variance of the phase of the     *
! *                                     sum of the signal and complex    *
! *                                     noise.                           *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! * ### 31-MAR-2010 PIMA_AMP_TO_SIGPHS v1.1 (d) L. Petrov 07-APR-2010 ## *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      INCLUDE   'astro_constants.i'
      INTEGER*4  DEG, MN, MS
      PARAMETER  ( DEG =   3 )
      PARAMETER  (  MN =   6 )
      PARAMETER  (  MS = 128 )
      REAL*8     SNR, AMP
      REAL*8     PIMA_AMP_TO_SIGPHS
!
      REAL*8     SPL_ARG(MN,MS), SPL_COE(1-DEG:MN-1,MS)
      REAL*8     SNR_MIN, SNR_MAX, SNR_STP, AMP_MIN
      PARAMETER  ( SNR_MIN =  0.0D0 )
      PARAMETER  ( SNR_MAX = 12.7D0 )
      PARAMETER  ( SNR_STP =  0.1D0 )
      PARAMETER  ( AMP_MIN = 1.D-10 )
      INTEGER*4  N1, N2
      DATA (SPL_ARG(N1,  1), N1=1,  6)  &  !  SNR=   0.000
     &     / &
     &      0.000000, & !  1
     &      0.914063, & !  2
     &      1.828125, & !  3
     &      2.742188, & !  4
     &      3.656250, & !  5
     &      4.570313  & !  6
     &     /
      DATA (SPL_COE(N2,  1), N2=-2,  5) &  !  SNR=   0.000
     &     / &
     &      1.813807, & ! -2
     &      1.813721, & ! -1
     &      1.813997, & !  0
     &      1.813481, & !  1
     &      1.814338, & !  2
     &      1.813154, & !  3
     &      1.814933, & !  4
     &      1.813037  & !  5
     &     /
!
      DATA (SPL_ARG(N1,  2), N1=1,  6)  &  !  SNR=   0.100
     &     / &
     &      0.000000, & !  1
     &      0.934375, & !  2
     &      1.868750, & !  3
     &      2.803125, & !  4
     &      3.737500, & !  5
     &      4.671875  & !  6
     &     /
      DATA (SPL_COE(N2,  2), N2=-2,  5) &  !  SNR=   0.100
     &     / &
     &      1.813803, & ! -2
     &      1.796768, & ! -1
     &      1.761828, & !  0
     &      1.709562, & !  1
     &      1.656249, & !  2
     &      1.604091, & !  3
     &      1.567439, & !  4
     &      1.551046  & !  5
     &     /
!
      DATA (SPL_ARG(N1,  3), N1=1,  6)  &  !  SNR=   0.200
     &     / &
     &      0.000000, & !  1
     &      0.954688, & !  2
     &      1.909375, & !  3
     &      2.864063, & !  4
     &      3.818750, & !  5
     &      4.773438  & !  6
     &     /
      DATA (SPL_COE(N2,  3), N2=-2,  5) &  !  SNR=   0.200
     &     / &
     &      1.813682, & ! -2
     &      1.778567, & ! -1
     &      1.707590, & !  0
     &      1.598485, & !  1
     &      1.492295, & !  2
     &      1.384683, & !  3
     &      1.327988, & !  4
     &      1.278438  & !  5
     &     /
!
      DATA (SPL_ARG(N1,  4), N1=1,  6)  &  !  SNR=   0.300
     &     / &
     &      0.000000, & !  1
     &      0.975000, & !  2
     &      1.950000, & !  3
     &      2.925000, & !  4
     &      3.900000, & !  5
     &      4.875000  & !  6
     &     /
      DATA (SPL_COE(N2,  4), N2=-2,  5) &  !  SNR=   0.300
     &     / &
     &      1.813541, & ! -2
     &      1.759664, & ! -1
     &      1.650485, & !  0
     &      1.482826, & !  1
     &      1.328159, & !  2
     &      1.178083, & !  3
     &      1.104764, & !  4
     &      1.047649  & !  5
     &     /
!
      DATA (SPL_ARG(N1,  5), N1=1,  6)  &  !  SNR=   0.400
     &     / &
     &      0.000000, & !  1
     &      0.995313, & !  2
     &      1.990625, & !  3
     &      2.985937, & !  4
     &      3.981250, & !  5
     &      4.976563  & !  6
     &     /
      DATA (SPL_COE(N2,  5), N2=-2,  5) &  !  SNR=   0.400
     &     / &
     &      1.813117, & ! -2
     &      1.740699, & ! -1
     &      1.589218, & !  0
     &      1.366432, & !  1
     &      1.169638, & !  2
     &      1.001099, & !  3
     &      0.920307, & !  4
     &      0.868620  & !  5
     &     /
!
      DATA (SPL_ARG(N1,  6), N1=1,  6)  &  !  SNR=   0.500
     &     / &
     &      0.000000, & !  1
     &      1.015625, & !  2
     &      2.031250, & !  3
     &      3.046875, & !  4
     &      4.062500, & !  5
     &      5.078125  & !  6
     &     /
      DATA (SPL_COE(N2,  6), N2=-2,  5) &  !  SNR=   0.500
     &     / &
     &      1.812778, & ! -2
     &      1.720875, & ! -1
     &      1.524917, & !  0
     &      1.250640, & !  1
     &      1.026011, & !  2
     &      0.857127, & !  3
     &      0.779551, & !  4
     &      0.730553  & !  5
     &     /
!
      DATA (SPL_ARG(N1,  7), N1=1,  6)  &  !  SNR=   0.600
     &     / &
     &      0.000000, & !  1
     &      1.035937, & !  2
     &      2.071875, & !  3
     &      3.107813, & !  4
     &      4.143750, & !  5
     &      5.179688  & !  6
     &     /
      DATA (SPL_COE(N2,  7), N2=-2,  5) &  !  SNR=   0.600
     &     / &
     &      1.812575, & ! -2
     &      1.699991, & ! -1
     &      1.458025, & !  0
     &      1.137536, & !  1
     &      0.903756, & !  2
     &      0.741451, & !  3
     &      0.679056, & !  4
     &      0.637986  & !  5
     &     /
!
      DATA (SPL_ARG(N1,  8), N1=1,  6)  &  !  SNR=   0.700
     &     / &
     &      0.000000, & !  1
     &      1.056250, & !  2
     &      2.112500, & !  3
     &      3.168750, & !  4
     &      4.225000, & !  5
     &      5.281250  & !  6
     &     /
      DATA (SPL_COE(N2,  8), N2=-2,  5) &  !  SNR=   0.700
     &     / &
     &      1.811868, & ! -2
     &      1.678781, & ! -1
     &      1.387531, & !  0
     &      1.031704, & !  1
     &      0.800989, & !  2
     &      0.656846, & !  3
     &      0.604590, & !  4
     &      0.562152  & !  5
     &     /
!
      DATA (SPL_ARG(N1,  9), N1=1,  6)  &  !  SNR=   0.800
     &     / &
     &      0.000000, & !  1
     &      1.076563, & !  2
     &      2.153125, & !  3
     &      3.229688, & !  4
     &      4.306250, & !  5
     &      5.382813  & !  6
     &     /
      DATA (SPL_COE(N2,  9), N2=-2,  5) &  !  SNR=   0.800
     &     / &
     &      1.811834, & ! -2
     &      1.656598, & ! -1
     &      1.313932, & !  0
     &      0.935480, & !  1
     &      0.718083, & !  2
     &      0.592715, & !  3
     &      0.545261, & !  4
     &      0.520565  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 10), N1=1,  6)  &  !  SNR=   0.900
     &     / &
     &      0.000000, & !  1
     &      1.096875, & !  2
     &      2.193750, & !  3
     &      3.290625, & !  4
     &      4.387500, & !  5
     &      5.484375  & !  6
     &     /
      DATA (SPL_COE(N2, 10), N2=-2,  5) &  !  SNR=   0.900
     &     / &
     &      1.811488, & ! -2
     &      1.633171, & ! -1
     &      1.238640, & !  0
     &      0.849638, & !  1
     &      0.652791, & !  2
     &      0.541709, & !  3
     &      0.502225, & !  4
     &      0.475112  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 11), N1=1,  6)  &  !  SNR=   1.000
     &     / &
     &      0.000000, & !  1
     &      1.117188, & !  2
     &      2.234375, & !  3
     &      3.351563, & !  4
     &      4.468750, & !  5
     &      5.585938  & !  6
     &     /
      DATA (SPL_COE(N2, 11), N2=-2,  5) &  !  SNR=   1.000
     &     / &
     &      1.811520, & ! -2
     &      1.608592, & ! -1
     &      1.161970, & !  0
     &      0.775968, & !  1
     &      0.600398, & !  2
     &      0.502498, & !  3
     &      0.465314, & !  4
     &      0.446452  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 12), N1=1,  6)  &  !  SNR=   1.100
     &     / &
     &      0.000000, & !  1
     &      1.137500, & !  2
     &      2.275000, & !  3
     &      3.412500, & !  4
     &      4.550000, & !  5
     &      5.687500  & !  6
     &     /
      DATA (SPL_COE(N2, 12), N2=-2,  5) &  !  SNR=   1.100
     &     / &
     &      1.811963, & ! -2
     &      1.581808, & ! -1
     &      1.085794, & !  0
     &      0.713716, & !  1
     &      0.558044, & !  2
     &      0.469262, & !  3
     &      0.437834, & !  4
     &      0.414310  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 13), N1=1,  6)  &  !  SNR=   1.200
     &     / &
     &      0.000000, & !  1
     &      1.157813, & !  2
     &      2.315625, & !  3
     &      3.473437, & !  4
     &      4.631250, & !  5
     &      5.789062  & !  6
     &     /
      DATA (SPL_COE(N2, 13), N2=-2,  5) &  !  SNR=   1.200
     &     / &
     &      1.812670, & ! -2
     &      1.552802, & ! -1
     &      1.011147, & !  0
     &      0.662252, & !  1
     &      0.522624, & !  2
     &      0.442486, & !  3
     &      0.411539, & !  4
     &      0.392468  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 14), N1=1,  6)  &  !  SNR=   1.300
     &     / &
     &      0.000000, & !  1
     &      1.178125, & !  2
     &      2.356250, & !  3
     &      3.534375, & !  4
     &      4.712500, & !  5
     &      5.890625  & !  6
     &     /
      DATA (SPL_COE(N2, 14), N2=-2,  5) &  !  SNR=   1.300
     &     / &
     &      1.813892, & ! -2
     &      1.520978, & ! -1
     &      0.939878, & !  0
     &      0.619677, & !  1
     &      0.492872, & !  2
     &      0.418426, & !  3
     &      0.392138, & !  4
     &      0.371551  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 15), N1=1,  6)  &  !  SNR=   1.400
     &     / &
     &      0.000000, & !  1
     &      1.198438, & !  2
     &      2.396875, & !  3
     &      3.595312, & !  4
     &      4.793750, & !  5
     &      5.992188  & !  6
     &     /
      DATA (SPL_COE(N2, 15), N2=-2,  5) &  !  SNR=   1.400
     &     / &
     &      1.816452, & ! -2
     &      1.486018, & ! -1
     &      0.872945, & !  0
     &      0.584960, & !  1
     &      0.466212, & !  2
     &      0.399209, & !  3
     &      0.370811, & !  4
     &      0.355722  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 16), N1=1,  6)  &  !  SNR=   1.500
     &     / &
     &      0.000000, & !  1
     &      1.218750, & !  2
     &      2.437500, & !  3
     &      3.656250, & !  4
     &      4.875000, & !  5
     &      6.093750  & !  6
     &     /
      DATA (SPL_COE(N2, 16), N2=-2,  5) &  !  SNR=   1.500
     &     / &
     &      1.818574, & ! -2
     &      1.448597, & ! -1
     &      0.810808, & !  0
     &      0.556411, & !  1
     &      0.442271, & !  2
     &      0.382454, & !  3
     &      0.352503, & !  4
     &      0.341027  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 17), N1=1,  6)  &  !  SNR=   1.600
     &     / &
     &      0.000000, & !  1
     &      1.239062, & !  2
     &      2.478125, & !  3
     &      3.717187, & !  4
     &      4.956250, & !  5
     &      6.195313  & !  6
     &     /
      DATA (SPL_COE(N2, 17), N2=-2,  5) &  !  SNR=   1.600
     &     / &
     &      1.821893, & ! -2
     &      1.407977, & ! -1
     &      0.754773, & !  0
     &      0.532149, & !  1
     &      0.421127, & !  2
     &      0.366960, & !  3
     &      0.337578, & !  4
     &      0.324787  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 18), N1=1,  6)  &  !  SNR=   1.700
     &     / &
     &      0.000000, & !  1
     &      1.259375, & !  2
     &      2.518750, & !  3
     &      3.778125, & !  4
     &      5.037500, & !  5
     &      6.296875  & !  6
     &     /
      DATA (SPL_COE(N2, 18), N2=-2,  5) &  !  SNR=   1.700
     &     / &
     &      1.825920, & ! -2
     &      1.364448, & ! -1
     &      0.704710, & !  0
     &      0.511455, & !  1
     &      0.401544, & !  2
     &      0.354011, & !  3
     &      0.322389, & !  4
     &      0.315736  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 19), N1=1,  6)  &  !  SNR=   1.800
     &     / &
     &      0.000000, & !  1
     &      1.279687, & !  2
     &      2.559375, & !  3
     &      3.839062, & !  4
     &      5.118750, & !  5
     &      6.398437  & !  6
     &     /
      DATA (SPL_COE(N2, 19), N2=-2,  5) &  !  SNR=   1.800
     &     / &
     &      1.829832, & ! -2
     &      1.319173, & ! -1
     &      0.660311, & !  0
     &      0.493159, & !  1
     &      0.384140, & !  2
     &      0.341532, & !  3
     &      0.309762, & !  4
     &      0.302526  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 20), N1=1,  6)  &  !  SNR=   1.900
     &     / &
     &      0.000000, & !  1
     &      1.300000, & !  2
     &      2.600000, & !  3
     &      3.900000, & !  4
     &      5.200000, & !  5
     &      6.500000  & !  6
     &     /
      DATA (SPL_COE(N2, 20), N2=-2,  5) &  !  SNR=   1.900
     &     / &
     &      1.833898, & ! -2
     &      1.272142, & ! -1
     &      0.621677, & !  0
     &      0.476720, & !  1
     &      0.368062, & !  2
     &      0.330845, & !  3
     &      0.296918, & !  4
     &      0.291918  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 21), N1=1,  6)  &  !  SNR=   2.000
     &     / &
     &      0.000000, & !  1
     &      1.320313, & !  2
     &      2.640625, & !  3
     &      3.960938, & !  4
     &      5.281250, & !  5
     &      6.601563  & !  6
     &     /
      DATA (SPL_COE(N2, 21), N2=-2,  5) &  !  SNR=   2.000
     &     / &
     &      1.838287, & ! -2
     &      1.223901, & ! -1
     &      0.588172, & !  0
     &      0.461432, & !  1
     &      0.353775, & !  2
     &      0.320149, & !  3
     &      0.286361, & !  4
     &      0.282348  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 22), N1=1,  6)  &  !  SNR=   2.100
     &     / &
     &      0.000000, & !  1
     &      1.340625, & !  2
     &      2.681250, & !  3
     &      4.021875, & !  4
     &      5.362500, & !  5
     &      6.703125  & !  6
     &     /
      DATA (SPL_COE(N2, 22), N2=-2,  5) &  !  SNR=   2.100
     &     / &
     &      1.840628, & ! -2
     &      1.175564, & ! -1
     &      0.559337, & !  0
     &      0.447093, & !  1
     &      0.340809, & !  2
     &      0.310214, & !  3
     &      0.276856, & !  4
     &      0.272118  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 23), N1=1,  6)  &  !  SNR=   2.200
     &     / &
     &      0.000000, & !  1
     &      1.360937, & !  2
     &      2.721875, & !  3
     &      4.082812, & !  4
     &      5.443750, & !  5
     &      6.804687  & !  6
     &     /
      DATA (SPL_COE(N2, 23), N2=-2,  5) &  !  SNR=   2.200
     &     / &
     &      1.843892, & ! -2
     &      1.126078, & ! -1
     &      0.535243, & !  0
     &      0.433070, & !  1
     &      0.329434, & !  2
     &      0.300282, & !  3
     &      0.268572, & !  4
     &      0.264386  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 24), N1=1,  6)  &  !  SNR=   2.300
     &     / &
     &      0.000000, & !  1
     &      1.381250, & !  2
     &      2.762500, & !  3
     &      4.143750, & !  4
     &      5.525000, & !  5
     &      6.906250  & !  6
     &     /
      DATA (SPL_COE(N2, 24), N2=-2,  5) &  !  SNR=   2.300
     &     / &
     &      1.846716, & ! -2
     &      1.077081, & ! -1
     &      0.514653, & !  0
     &      0.419865, & !  1
     &      0.318675, & !  2
     &      0.291927, & !  3
     &      0.258829, & !  4
     &      0.257045  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 25), N1=1,  6)  &  !  SNR=   2.400
     &     / &
     &      0.000000, & !  1
     &      1.401562, & !  2
     &      2.803125, & !  3
     &      4.204688, & !  4
     &      5.606250, & !  5
     &      7.007813  & !  6
     &     /
      DATA (SPL_COE(N2, 25), N2=-2,  5) &  !  SNR=   2.400
     &     / &
     &      1.847951, & ! -2
     &      1.028806, & ! -1
     &      0.497375, & !  0
     &      0.406874, & !  1
     &      0.309331, & !  2
     &      0.283089, & !  3
     &      0.251898, & !  4
     &      0.248555  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 26), N1=1,  6)  &  !  SNR=   2.500
     &     / &
     &      0.000000, & !  1
     &      1.421875, & !  2
     &      2.843750, & !  3
     &      4.265625, & !  4
     &      5.687500, & !  5
     &      7.109375  & !  6
     &     /
      DATA (SPL_COE(N2, 26), N2=-2,  5) &  !  SNR=   2.500
     &     / &
     &      1.848755, & ! -2
     &      0.981376, & ! -1
     &      0.482994, & !  0
     &      0.394156, & !  1
     &      0.300878, & !  2
     &      0.274563, & !  3
     &      0.245844, & !  4
     &      0.241549  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 27), N1=1,  6)  &  !  SNR=   2.600
     &     / &
     &      0.000000, & !  1
     &      1.442187, & !  2
     &      2.884375, & !  3
     &      4.326562, & !  4
     &      5.768750, & !  5
     &      7.210938  & !  6
     &     /
      DATA (SPL_COE(N2, 27), N2=-2,  5) &  !  SNR=   2.600
     &     / &
     &      1.848360, & ! -2
     &      0.934979, & ! -1
     &      0.470991, & !  0
     &      0.381778, & !  1
     &      0.293247, & !  2
     &      0.266577, & !  3
     &      0.239360, & !  4
     &      0.234016  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 28), N1=1,  6)  &  !  SNR=   2.700
     &     / &
     &      0.000000, & !  1
     &      1.462500, & !  2
     &      2.925000, & !  3
     &      4.387500, & !  4
     &      5.850000, & !  5
     &      7.312500  & !  6
     &     /
      DATA (SPL_COE(N2, 28), N2=-2,  5) &  !  SNR=   2.700
     &     / &
     &      1.849094, & ! -2
     &      0.888839, & ! -1
     &      0.461660, & !  0
     &      0.369388, & !  1
     &      0.286511, & !  2
     &      0.258633, & !  3
     &      0.233544, & !  4
     &      0.228298  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 29), N1=1,  6)  &  !  SNR=   2.800
     &     / &
     &      0.000000, & !  1
     &      1.482813, & !  2
     &      2.965625, & !  3
     &      4.448438, & !  4
     &      5.931250, & !  5
     &      7.414063  & !  6
     &     /
      DATA (SPL_COE(N2, 29), N2=-2,  5) &  !  SNR=   2.800
     &     / &
     &      1.844216, & ! -2
     &      0.846336, & ! -1
     &      0.452831, & !  0
     &      0.357883, & !  1
     &      0.280090, & !  2
     &      0.251265, & !  3
     &      0.228302, & !  4
     &      0.221464  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 30), N1=1,  6)  &  !  SNR=   2.900
     &     / &
     &      0.000000, & !  1
     &      1.503125, & !  2
     &      3.006250, & !  3
     &      4.509375, & !  4
     &      6.012500, & !  5
     &      7.515625  & !  6
     &     /
      DATA (SPL_COE(N2, 30), N2=-2,  5) &  !  SNR=   2.900
     &     / &
     &      1.842069, & ! -2
     &      0.803555, & ! -1
     &      0.446427, & !  0
     &      0.346163, & !  1
     &      0.274605, & !  2
     &      0.243698, & !  3
     &      0.223560, & !  4
     &      0.215535  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 31), N1=1,  6)  &  !  SNR=   3.000
     &     / &
     &      0.000000, & !  1
     &      1.523438, & !  2
     &      3.046875, & !  3
     &      4.570313, & !  4
     &      6.093750, & !  5
     &      7.617187  & !  6
     &     /
      DATA (SPL_COE(N2, 31), N2=-2,  5) &  !  SNR=   3.000
     &     / &
     &      1.838564, & ! -2
     &      0.762740, & ! -1
     &      0.440993, & !  0
     &      0.334934, & !  1
     &      0.269365, & !  2
     &      0.236932, & !  3
     &      0.217998, & !  4
     &      0.211660  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 32), N1=1,  6)  &  !  SNR=   3.100
     &     / &
     &      0.000000, & !  1
     &      1.543750, & !  2
     &      3.087500, & !  3
     &      4.631250, & !  4
     &      6.175000, & !  5
     &      7.718750  & !  6
     &     /
      DATA (SPL_COE(N2, 32), N2=-2,  5) &  !  SNR=   3.100
     &     / &
     &      1.833783, & ! -2
     &      0.723550, & ! -1
     &      0.436594, & !  0
     &      0.324036, & !  1
     &      0.264511, & !  2
     &      0.230390, & !  3
     &      0.213344, & !  4
     &      0.206302  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 33), N1=1,  6)  &  !  SNR=   3.200
     &     / &
     &      0.000000, & !  1
     &      1.564062, & !  2
     &      3.128125, & !  3
     &      4.692187, & !  4
     &      6.256250, & !  5
     &      7.820313  & !  6
     &     /
      DATA (SPL_COE(N2, 33), N2=-2,  5) &  !  SNR=   3.200
     &     / &
     &      1.828125, & ! -2
     &      0.686652, & ! -1
     &      0.432664, & !  0
     &      0.313444, & !  1
     &      0.260347, & !  2
     &      0.223279, & !  3
     &      0.210339, & !  4
     &      0.200543  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 34), N1=1,  6)  &  !  SNR=   3.300
     &     / &
     &      0.000000, & !  1
     &      1.584375, & !  2
     &      3.168750, & !  3
     &      4.753125, & !  4
     &      6.337500, & !  5
     &      7.921875  & !  6
     &     /
      DATA (SPL_COE(N2, 34), N2=-2,  5) &  !  SNR=   3.300
     &     / &
     &      1.820209, & ! -2
     &      0.651803, & ! -1
     &      0.429172, & !  0
     &      0.303361, & !  1
     &      0.256252, & !  2
     &      0.216958, & !  3
     &      0.206714, & !  4
     &      0.194904  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 35), N1=1,  6)  &  !  SNR=   3.400
     &     / &
     &      0.000000, & !  1
     &      1.604688, & !  2
     &      3.209375, & !  3
     &      4.814063, & !  4
     &      6.418750, & !  5
     &      8.023438  & !  6
     &     /
      DATA (SPL_COE(N2, 35), N2=-2,  5) &  !  SNR=   3.400
     &     / &
     &      1.813531, & ! -2
     &      0.616711, & ! -1
     &      0.427372, & !  0
     &      0.293082, & !  1
     &      0.252706, & !  2
     &      0.210692, & !  3
     &      0.203266, & !  4
     &      0.190954  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 36), N1=1,  6)  &  !  SNR=   3.500
     &     / &
     &      0.000000, & !  1
     &      1.625000, & !  2
     &      3.250000, & !  3
     &      4.875000, & !  4
     &      6.500000, & !  5
     &      8.125000  & !  6
     &     /
      DATA (SPL_COE(N2, 36), N2=-2,  5) &  !  SNR=   3.500
     &     / &
     &      1.804567, & ! -2
     &      0.586258, & ! -1
     &      0.424034, & !  0
     &      0.284115, & !  1
     &      0.248794, & !  2
     &      0.205342, & !  3
     &      0.198943, & !  4
     &      0.188289  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 37), N1=1,  6)  &  !  SNR=   3.600
     &     / &
     &      0.000000, & !  1
     &      1.645312, & !  2
     &      3.290625, & !  3
     &      4.935937, & !  4
     &      6.581250, & !  5
     &      8.226563  & !  6
     &     /
      DATA (SPL_COE(N2, 37), N2=-2,  5) &  !  SNR=   3.600
     &     / &
     &      1.802010, & ! -2
     &      0.550748, & ! -1
     &      0.424662, & !  0
     &      0.273812, & !  1
     &      0.246026, & !  2
     &      0.199525, & !  3
     &      0.195708, & !  4
     &      0.181949  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 38), N1=1,  6)  &  !  SNR=   3.700
     &     / &
     &      0.000000, & !  1
     &      1.665625, & !  2
     &      3.331250, & !  3
     &      4.996875, & !  4
     &      6.662500, & !  5
     &      8.328125  & !  6
     &     /
      DATA (SPL_COE(N2, 38), N2=-2,  5) &  !  SNR=   3.700
     &     / &
     &      1.790065, & ! -2
     &      0.522123, & ! -1
     &      0.422888, & !  0
     &      0.264871, & !  1
     &      0.243230, & !  2
     &      0.193294, & !  3
     &      0.194089, & !  4
     &      0.176689  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 39), N1=1,  6)  &  !  SNR=   3.800
     &     / &
     &      0.000000, & !  1
     &      1.685938, & !  2
     &      3.371875, & !  3
     &      5.057812, & !  4
     &      6.743750, & !  5
     &      8.429688  & !  6
     &     /
      DATA (SPL_COE(N2, 39), N2=-2,  5) &  !  SNR=   3.800
     &     / &
     &      1.773944, & ! -2
     &      0.498368, & ! -1
     &      0.419355, & !  0
     &      0.257360, & !  1
     &      0.239562, & !  2
     &      0.188907, & !  3
     &      0.189953, & !  4
     &      0.174026  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 40), N1=1,  6)  &  !  SNR=   3.900
     &     / &
     &      0.000000, & !  1
     &      1.706250, & !  2
     &      3.412500, & !  3
     &      5.118750, & !  4
     &      6.825000, & !  5
     &      8.531250  & !  6
     &     /
      DATA (SPL_COE(N2, 40), N2=-2,  5) &  !  SNR=   3.900
     &     / &
     &      1.767111, & ! -2
     &      0.471167, & ! -1
     &      0.418182, & !  0
     &      0.249038, & !  1
     &      0.236965, & !  2
     &      0.183441, & !  3
     &      0.187983, & !  4
     &      0.170389  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 41), N1=1,  6)  &  !  SNR=   4.000
     &     / &
     &      0.000000, & !  1
     &      1.726563, & !  2
     &      3.453125, & !  3
     &      5.179687, & !  4
     &      6.906250, & !  5
     &      8.632813  & !  6
     &     /
      DATA (SPL_COE(N2, 41), N2=-2,  5) &  !  SNR=   4.000
     &     / &
     &      1.755982, & ! -2
     &      0.444256, & ! -1
     &      0.418215, & !  0
     &      0.240579, & !  1
     &      0.234541, & !  2
     &      0.178619, & !  3
     &      0.184825, & !  4
     &      0.168055  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 42), N1=1,  6)  &  !  SNR=   4.100
     &     / &
     &      0.000000, & !  1
     &      1.746875, & !  2
     &      3.493750, & !  3
     &      5.240625, & !  4
     &      6.987500, & !  5
     &      8.734375  & !  6
     &     /
      DATA (SPL_COE(N2, 42), N2=-2,  5) &  !  SNR=   4.100
     &     / &
     &      1.736285, & ! -2
     &      0.425320, & ! -1
     &      0.414617, & !  0
     &      0.234071, & !  1
     &      0.231507, & !  2
     &      0.174004, & !  3
     &      0.182937, & !  4
     &      0.162721  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 43), N1=1,  6)  &  !  SNR=   4.200
     &     / &
     &      0.000000, & !  1
     &      1.767187, & !  2
     &      3.534375, & !  3
     &      5.301563, & !  4
     &      7.068750, & !  5
     &      8.835938  & !  6
     &     /
      DATA (SPL_COE(N2, 43), N2=-2,  5) &  !  SNR=   4.200
     &     / &
     &      1.742288, & ! -2
     &      0.392592, & ! -1
     &      0.418895, & !  0
     &      0.224279, & !  1
     &      0.230477, & !  2
     &      0.168566, & !  3
     &      0.180719, & !  4
     &      0.158803  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 44), N1=1,  6)  &  !  SNR=   4.300
     &     / &
     &      0.000000, & !  1
     &      1.787500, & !  2
     &      3.575000, & !  3
     &      5.362500, & !  4
     &      7.150000, & !  5
     &      8.937500  & !  6
     &     /
      DATA (SPL_COE(N2, 44), N2=-2,  5) &  !  SNR=   4.300
     &     / &
     &      1.731106, & ! -2
     &      0.371974, & ! -1
     &      0.417397, & !  0
     &      0.217458, & !  1
     &      0.228092, & !  2
     &      0.164103, & !  3
     &      0.178932, & !  4
     &      0.155847  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 45), N1=1,  6)  &  !  SNR=   4.400
     &     / &
     &      0.000000, & !  1
     &      1.807812, & !  2
     &      3.615625, & !  3
     &      5.423438, & !  4
     &      7.231250, & !  5
     &      9.039063  & !  6
     &     /
      DATA (SPL_COE(N2, 45), N2=-2,  5) &  !  SNR=   4.400
     &     / &
     &      1.747592, & ! -2
     &      0.336789, & ! -1
     &      0.423973, & !  0
     &      0.207163, & !  1
     &      0.227915, & !  2
     &      0.158274, & !  3
     &      0.177763, & !  4
     &      0.152618  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 46), N1=1,  6)  &  !  SNR=   4.500
     &     / &
     &      0.000000, & !  1
     &      1.828125, & !  2
     &      3.656250, & !  3
     &      5.484375, & !  4
     &      7.312500, & !  5
     &      9.140625  & !  6
     &     /
      DATA (SPL_COE(N2, 46), N2=-2,  5) &  !  SNR=   4.500
     &     / &
     &      1.682252, & ! -2
     &      0.340590, & ! -1
     &      0.410861, & !  0
     &      0.206224, & !  1
     &      0.222569, & !  2
     &      0.156555, & !  3
     &      0.174371, & !  4
     &      0.148617  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 47), N1=1,  6)  &  !  SNR=   4.600
     &     / &
     &      0.100000, & !  1
     &      1.928125, & !  2
     &      3.756250, & !  3
     &      5.584375, & !  4
     &      7.412500, & !  5
     &      9.240625  & !  6
     &     /
      DATA (SPL_COE(N2, 47), N2=-2,  5) &  !  SNR=   4.600
     &     / &
     &      1.457138, & ! -2
     &      0.307881, & ! -1
     &      0.405813, & !  0
     &      0.197503, & !  1
     &      0.220411, & !  2
     &      0.151996, & !  3
     &      0.172688, & !  4
     &      0.145924  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 48), N1=1,  6)  &  !  SNR=   4.700
     &     / &
     &      0.200000, & !  1
     &      2.028125, & !  2
     &      3.856250, & !  3
     &      5.684375, & !  4
     &      7.512500, & !  5
     &      9.340625  & !  6
     &     /
      DATA (SPL_COE(N2, 48), N2=-2,  5) &  !  SNR=   4.700
     &     / &
     &      1.228867, & ! -2
     &      0.305657, & ! -1
     &      0.386293, & !  0
     &      0.196272, & !  1
     &      0.213983, & !  2
     &      0.151495, & !  3
     &      0.167675, & !  4
     &      0.146330  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 49), N1=1,  6)  &  !  SNR=   4.800
     &     / &
     &      0.300000, & !  1
     &      2.128125, & !  2
     &      3.956250, & !  3
     &      5.784375, & !  4
     &      7.612500, & !  5
     &      9.440625  & !  6
     &     /
      DATA (SPL_COE(N2, 49), N2=-2,  5) &  !  SNR=   4.800
     &     / &
     &      1.021578, & ! -2
     &      0.321773, & ! -1
     &      0.358326, & !  0
     &      0.199398, & !  1
     &      0.205518, & !  2
     &      0.152245, & !  3
     &      0.163308, & !  4
     &      0.144128  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 50), N1=1,  6)  &  !  SNR=   4.900
     &     / &
     &      0.400000, & !  1
     &      2.228125, & !  2
     &      4.056250, & !  3
     &      5.884375, & !  4
     &      7.712500, & !  5
     &      9.540625  & !  6
     &     /
      DATA (SPL_COE(N2, 50), N2=-2,  5) &  !  SNR=   4.900
     &     / &
     &      0.876820, & ! -2
     &      0.330467, & ! -1
     &      0.334487, & !  0
     &      0.201373, & !  1
     &      0.197838, & !  2
     &      0.152937, & !  3
     &      0.158217, & !  4
     &      0.143567  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 51), N1=1,  6)  &  !  SNR=   5.000
     &     / &
     &      0.500000, & !  1
     &      2.328125, & !  2
     &      4.156250, & !  3
     &      5.984375, & !  4
     &      7.812500, & !  5
     &      9.640625  & !  6
     &     /
      DATA (SPL_COE(N2, 51), N2=-2,  5) &  !  SNR=   5.000
     &     / &
     &      0.763358, & ! -2
     &      0.331555, & ! -1
     &      0.316841, & !  0
     &      0.200779, & !  1
     &      0.192111, & !  2
     &      0.151842, & !  3
     &      0.155349, & !  4
     &      0.140869  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 52), N1=1,  6)  &  !  SNR=   5.100
     &     / &
     &      0.600000, & !  1
     &      2.428125, & !  2
     &      4.256250, & !  3
     &      6.084375, & !  4
     &      7.912500, & !  5
     &      9.740625  & !  6
     &     /
      DATA (SPL_COE(N2, 52), N2=-2,  5) &  !  SNR=   5.100
     &     / &
     &      0.674302, & ! -2
     &      0.330658, & ! -1
     &      0.300715, & !  0
     &      0.200050, & !  1
     &      0.186547, & !  2
     &      0.151136, & !  3
     &      0.151809, & !  4
     &      0.139824  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 53), N1=1,  6)  &  !  SNR=   5.200
     &     / &
     &      0.700000, & !  1
     &      2.528125, & !  2
     &      4.356250, & !  3
     &      6.184375, & !  4
     &      8.012500, & !  5
     &      9.840625  & !  6
     &     /
      DATA (SPL_COE(N2, 53), N2=-2,  5) &  !  SNR=   5.200
     &     / &
     &      0.611161, & ! -2
     &      0.323773, & ! -1
     &      0.288795, & !  0
     &      0.197772, & !  1
     &      0.182091, & !  2
     &      0.149725, & !  3
     &      0.148650, & !  4
     &      0.137041  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 54), N1=1,  6)  &  !  SNR=   5.300
     &     / &
     &      0.800000, & !  1
     &      2.628125, & !  2
     &      4.456250, & !  3
     &      6.284375, & !  4
     &      8.112500, & !  5
     &      9.940625  & !  6
     &     /
      DATA (SPL_COE(N2, 54), N2=-2,  5) &  !  SNR=   5.300
     &     / &
     &      0.553015, & ! -2
     &      0.320648, & ! -1
     &      0.275845, & !  0
     &      0.196371, & !  1
     &      0.177273, & !  2
     &      0.148970, & !  3
     &      0.145104, & !  4
     &      0.136792  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 55), N1=1,  6)  &  !  SNR=   5.400
     &     / &
     &      0.900000, & !  1
     &      2.728125, & !  2
     &      4.556250, & !  3
     &      6.384375, & !  4
     &      8.212500, & !  5
     &     10.040625  & !  6
     &     /
      DATA (SPL_COE(N2, 55), N2=-2,  5) &  !  SNR=   5.400
     &     / &
     &      0.508976, & ! -2
     &      0.314324, & ! -1
     &      0.265645, & !  0
     &      0.193955, & !  1
     &      0.173374, & !  2
     &      0.147352, & !  3
     &      0.142360, & !  4
     &      0.135136  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 56), N1=1,  6)  &  !  SNR=   5.500
     &     / &
     &      1.000000, & !  1
     &      2.828125, & !  2
     &      4.656250, & !  3
     &      6.484375, & !  4
     &      8.312500, & !  5
     &     10.140625  & !  6
     &     /
      DATA (SPL_COE(N2, 56), N2=-2,  5) &  !  SNR=   5.500
     &     / &
     &      0.470192, & ! -2
     &      0.307819, & ! -1
     &      0.256296, & !  0
     &      0.191448, & !  1
     &      0.169771, & !  2
     &      0.145551, & !  3
     &      0.140347, & !  4
     &      0.132583  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 57), N1=1,  6)  &  !  SNR=   5.600
     &     / &
     &      1.100000, & !  1
     &      2.928125, & !  2
     &      4.756250, & !  3
     &      6.584375, & !  4
     &      8.412500, & !  5
     &     10.240625  & !  6
     &     /
      DATA (SPL_COE(N2, 57), N2=-2,  5) &  !  SNR=   5.600
     &     / &
     &      0.438256, & ! -2
     &      0.300724, & ! -1
     &      0.247934, & !  0
     &      0.188836, & !  1
     &      0.166287, & !  2
     &      0.144068, & !  3
     &      0.137562, & !  4
     &      0.132054  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 58), N1=1,  6)  &  !  SNR=   5.700
     &     / &
     &      1.200000, & !  1
     &      3.028125, & !  2
     &      4.856250, & !  3
     &      6.684375, & !  4
     &      8.512500, & !  5
     &     10.340625  & !  6
     &     /
      DATA (SPL_COE(N2, 58), N2=-2,  5) &  !  SNR=   5.700
     &     / &
     &      0.412951, & ! -2
     &      0.292819, & ! -1
     &      0.240939, & !  0
     &      0.185776, & !  1
     &      0.163382, & !  2
     &      0.141954, & !  3
     &      0.135977, & !  4
     &      0.130180  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 59), N1=1,  6)  &  !  SNR=   5.800
     &     / &
     &      1.300000, & !  1
     &      3.128125, & !  2
     &      4.956250, & !  3
     &      6.784375, & !  4
     &      8.612500, & !  5
     &     10.440625  & !  6
     &     /
      DATA (SPL_COE(N2, 59), N2=-2,  5) &  !  SNR=   5.800
     &     / &
     &      0.389632, & ! -2
     &      0.284769, & ! -1
     &      0.234541, & !  0
     &      0.182774, & !  1
     &      0.160485, & !  2
     &      0.140222, & !  3
     &      0.133757, & !  4
     &      0.128953  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 60), N1=1,  6)  &  !  SNR=   5.900
     &     / &
     &      1.400000, & !  1
     &      3.228125, & !  2
     &      5.056250, & !  3
     &      6.884375, & !  4
     &      8.712500, & !  5
     &     10.540625  & !  6
     &     /
      DATA (SPL_COE(N2, 60), N2=-2,  5) &  !  SNR=   5.900
     &     / &
     &      0.369595, & ! -2
     &      0.277914, & ! -1
     &      0.228131, & !  0
     &      0.179976, & !  1
     &      0.157660, & !  2
     &      0.138491, & !  3
     &      0.131707, & !  4
     &      0.127027  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 61), N1=1,  6)  &  !  SNR=   6.000
     &     / &
     &      1.500000, & !  1
     &      3.328125, & !  2
     &      5.156250, & !  3
     &      6.984375, & !  4
     &      8.812500, & !  5
     &     10.640625  & !  6
     &     /
      DATA (SPL_COE(N2, 61), N2=-2,  5) &  !  SNR=   6.000
     &     / &
     &      0.351122, & ! -2
     &      0.269497, & ! -1
     &      0.223040, & !  0
     &      0.176795, & !  1
     &      0.155221, & !  2
     &      0.136508, & !  3
     &      0.130069, & !  4
     &      0.124216  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 62), N1=1,  6)  &  !  SNR=   6.100
     &     / &
     &      1.600000, & !  1
     &      3.428125, & !  2
     &      5.256250, & !  3
     &      7.084375, & !  4
     &      8.912500, & !  5
     &     10.740625  & !  6
     &     /
      DATA (SPL_COE(N2, 62), N2=-2,  5) &  !  SNR=   6.100
     &     / &
     &      0.335081, & ! -2
     &      0.263687, & ! -1
     &      0.217084, & !  0
     &      0.174243, & !  1
     &      0.152520, & !  2
     &      0.134922, & !  3
     &      0.128098, & !  4
     &      0.123540  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 63), N1=1,  6)  &  !  SNR=   6.200
     &     / &
     &      1.700000, & !  1
     &      3.528125, & !  2
     &      5.356250, & !  3
     &      7.184375, & !  4
     &      9.012500, & !  5
     &     10.840625  & !  6
     &     /
      DATA (SPL_COE(N2, 63), N2=-2,  5) &  !  SNR=   6.200
     &     / &
     &      0.321971, & ! -2
     &      0.255673, & ! -1
     &      0.212702, & !  0
     &      0.171115, & !  1
     &      0.150358, & !  2
     &      0.132926, & !  3
     &      0.126735, & !  4
     &      0.120579  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 64), N1=1,  6)  &  !  SNR=   6.300
     &     / &
     &      1.800000, & !  1
     &      3.628125, & !  2
     &      5.456250, & !  3
     &      7.284375, & !  4
     &      9.112500, & !  5
     &     10.940625  & !  6
     &     /
      DATA (SPL_COE(N2, 64), N2=-2,  5) &  !  SNR=   6.300
     &     / &
     &      0.309229, & ! -2
     &      0.249698, & ! -1
     &      0.207705, & !  0
     &      0.168501, & !  1
     &      0.147984, & !  2
     &      0.131206, & !  3
     &      0.125133, & !  4
     &      0.120261  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 65), N1=1,  6)  &  !  SNR=   6.400
     &     / &
     &      1.900000, & !  1
     &      3.728125, & !  2
     &      5.556250, & !  3
     &      7.384375, & !  4
     &      9.212500, & !  5
     &     11.040625  & !  6
     &     /
      DATA (SPL_COE(N2, 65), N2=-2,  5) &  !  SNR=   6.400
     &     / &
     &      0.298446, & ! -2
     &      0.243692, & ! -1
     &      0.203105, & !  0
     &      0.165909, & !  1
     &      0.145671, & !  2
     &      0.129696, & !  3
     &      0.123172, & !  4
     &      0.118475  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 66), N1=1,  6)  &  !  SNR=   6.500
     &     / &
     &      2.000000, & !  1
     &      3.828125, & !  2
     &      5.656250, & !  3
     &      7.484375, & !  4
     &      9.312500, & !  5
     &     11.140625  & !  6
     &     /
      DATA (SPL_COE(N2, 66), N2=-2,  5) &  !  SNR=   6.500
     &     / &
     &      0.289014, & ! -2
     &      0.236772, & ! -1
     &      0.199420, & !  0
     &      0.163005, & !  1
     &      0.143714, & !  2
     &      0.127872, & !  3
     &      0.121779, & !  4
     &      0.117245  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 67), N1=1,  6)  &  !  SNR=   6.600
     &     / &
     &      2.100000, & !  1
     &      3.928125, & !  2
     &      5.756250, & !  3
     &      7.584375, & !  4
     &      9.412500, & !  5
     &     11.240625  & !  6
     &     /
      DATA (SPL_COE(N2, 67), N2=-2,  5) &  !  SNR=   6.600
     &     / &
     &      0.277011, & ! -2
     &      0.232686, & ! -1
     &      0.194415, & !  0
     &      0.160915, & !  1
     &      0.141348, & !  2
     &      0.126524, & !  3
     &      0.119857, & !  4
     &      0.115920  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 68), N1=1,  6)  &  !  SNR=   6.700
     &     / &
     &      2.200000, & !  1
     &      4.028125, & !  2
     &      5.856250, & !  3
     &      7.684375, & !  4
     &      9.512500, & !  5
     &     11.340625  & !  6
     &     /
      DATA (SPL_COE(N2, 68), N2=-2,  5) &  !  SNR=   6.700
     &     / &
     &      0.268756, & ! -2
     &      0.225750, & ! -1
     &      0.191397, & !  0
     &      0.158050, & !  1
     &      0.139550, & !  2
     &      0.124787, & !  3
     &      0.118643, & !  4
     &      0.115168  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 69), N1=1,  6)  &  !  SNR=   6.800
     &     / &
     &      2.300000, & !  1
     &      4.128125, & !  2
     &      5.956250, & !  3
     &      7.784375, & !  4
     &      9.612500, & !  5
     &     11.440625  & !  6
     &     /
      DATA (SPL_COE(N2, 69), N2=-2,  5) &  !  SNR=   6.800
     &     / &
     &      0.262123, & ! -2
     &      0.220344, & ! -1
     &      0.187781, & !  0
     &      0.155576, & !  1
     &      0.137690, & !  2
     &      0.123088, & !  3
     &      0.117429, & !  4
     &      0.112199  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 70), N1=1,  6)  &  !  SNR=   6.900
     &     / &
     &      2.400000, & !  1
     &      4.228125, & !  2
     &      6.056250, & !  3
     &      7.884375, & !  4
     &      9.712500, & !  5
     &     11.540625  & !  6
     &     /
      DATA (SPL_COE(N2, 70), N2=-2,  5) &  !  SNR=   6.900
     &     / &
     &      0.252557, & ! -2
     &      0.216345, & ! -1
     &      0.183725, & !  0
     &      0.153491, & !  1
     &      0.135609, & !  2
     &      0.121791, & !  3
     &      0.115722, & !  4
     &      0.111758  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 71), N1=1,  6)  &  !  SNR=   7.000
     &     / &
     &      2.500000, & !  1
     &      4.328125, & !  2
     &      6.156250, & !  3
     &      7.984375, & !  4
     &      9.812500, & !  5
     &     11.640625  & !  6
     &     /
      DATA (SPL_COE(N2, 71), N2=-2,  5) &  !  SNR=   7.000
     &     / &
     &      0.246848, & ! -2
     &      0.210362, & ! -1
     &      0.180983, & !  0
     &      0.150878, & !  1
     &      0.133978, & !  2
     &      0.120166, & !  3
     &      0.114495, & !  4
     &      0.109983  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 72), N1=1,  6)  &  !  SNR=   7.100
     &     / &
     &      2.600000, & !  1
     &      4.428125, & !  2
     &      6.256250, & !  3
     &      8.084375, & !  4
     &      9.912500, & !  5
     &     11.740625  & !  6
     &     /
      DATA (SPL_COE(N2, 72), N2=-2,  5) &  !  SNR=   7.100
     &     / &
     &      0.241324, & ! -2
     &      0.204926, & ! -1
     &      0.178166, & !  0
     &      0.148406, & !  1
     &      0.132426, & !  2
     &      0.118356, & !  3
     &      0.113786, & !  4
     &      0.107853  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 73), N1=1,  6)  &  !  SNR=   7.200
     &     / &
     &      2.700000, & !  1
     &      4.528125, & !  2
     &      6.356250, & !  3
     &      8.184375, & !  4
     &     10.012500, & !  5
     &     11.840625  & !  6
     &     /
      DATA (SPL_COE(N2, 73), N2=-2,  5) &  !  SNR=   7.200
     &     / &
     &      0.232256, & ! -2
     &      0.202353, & ! -1
     &      0.173948, & !  0
     &      0.146783, & !  1
     &      0.130298, & !  2
     &      0.117410, & !  3
     &      0.111876, & !  4
     &      0.107598  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 74), N1=1,  6)  &  !  SNR=   7.300
     &     / &
     &      2.800000, & !  1
     &      4.628125, & !  2
     &      6.456250, & !  3
     &      8.284375, & !  4
     &     10.112500, & !  5
     &     11.940625  & !  6
     &     /
      DATA (SPL_COE(N2, 74), N2=-2,  5) &  !  SNR=   7.300
     &     / &
     &      0.225257, & ! -2
     &      0.197935, & ! -1
     &      0.171132, & !  0
     &      0.144597, & !  1
     &      0.128619, & !  2
     &      0.116123, & !  3
     &      0.110292, & !  4
     &      0.106988  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 75), N1=1,  6)  &  !  SNR=   7.400
     &     / &
     &      2.900000, & !  1
     &      4.728125, & !  2
     &      6.556250, & !  3
     &      8.384375, & !  4
     &     10.212500, & !  5
     &     12.040625  & !  6
     &     /
      DATA (SPL_COE(N2, 75), N2=-2,  5) &  !  SNR=   7.400
     &     / &
     &      0.221589, & ! -2
     &      0.193208, & ! -1
     &      0.168568, & !  0
     &      0.142377, & !  1
     &      0.127097, & !  2
     &      0.114626, & !  3
     &      0.109251, & !  4
     &      0.106339  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 76), N1=1,  6)  &  !  SNR=   7.500
     &     / &
     &      3.000000, & !  1
     &      4.828125, & !  2
     &      6.656250, & !  3
     &      8.484375, & !  4
     &     10.312500, & !  5
     &     12.140625  & !  6
     &     /
      DATA (SPL_COE(N2, 76), N2=-2,  5) &  !  SNR=   7.500
     &     / &
     &      0.215575, & ! -2
     &      0.189993, & ! -1
     &      0.165461, & !  0
     &      0.140478, & !  1
     &      0.125509, & !  2
     &      0.113179, & !  3
     &      0.108367, & !  4
     &      0.104360  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 77), N1=1,  6)  &  !  SNR=   7.600
     &     / &
     &      3.100000, & !  1
     &      4.928125, & !  2
     &      6.756250, & !  3
     &      8.584375, & !  4
     &     10.412500, & !  5
     &     12.240625  & !  6
     &     /
      DATA (SPL_COE(N2, 77), N2=-2,  5) &  !  SNR=   7.600
     &     / &
     &      0.209885, & ! -2
     &      0.186555, & ! -1
     &      0.162513, & !  0
     &      0.138676, & !  1
     &      0.123839, & !  2
     &      0.112060, & !  3
     &      0.106952, & !  4
     &      0.102832  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 78), N1=1,  6)  &  !  SNR=   7.700
     &     / &
     &      3.200000, & !  1
     &      5.028125, & !  2
     &      6.856250, & !  3
     &      8.684375, & !  4
     &     10.512500, & !  5
     &     12.340625  & !  6
     &     /
      DATA (SPL_COE(N2, 78), N2=-2,  5) &  !  SNR=   7.700
     &     / &
     &      0.205671, & ! -2
     &      0.182831, & ! -1
     &      0.159940, & !  0
     &      0.136784, & !  1
     &      0.122294, & !  2
     &      0.110899, & !  3
     &      0.105590, & !  4
     &      0.102256  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 79), N1=1,  6)  &  !  SNR=   7.800
     &     / &
     &      3.300000, & !  1
     &      5.128125, & !  2
     &      6.956250, & !  3
     &      8.784375, & !  4
     &     10.612500, & !  5
     &     12.440625  & !  6
     &     /
      DATA (SPL_COE(N2, 79), N2=-2,  5) &  !  SNR=   7.800
     &     / &
     &      0.200055, & ! -2
     &      0.179680, & ! -1
     &      0.157252, & !  0
     &      0.134992, & !  1
     &      0.120798, & !  2
     &      0.109648, & !  3
     &      0.104479, & !  4
     &      0.101265  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 80), N1=1,  6)  &  !  SNR=   7.900
     &     / &
     &      3.400000, & !  1
     &      5.228125, & !  2
     &      7.056250, & !  3
     &      8.884375, & !  4
     &     10.712500, & !  5
     &     12.540625  & !  6
     &     /
      DATA (SPL_COE(N2, 80), N2=-2,  5) &  !  SNR=   7.900
     &     / &
     &      0.195339, & ! -2
     &      0.176856, & ! -1
     &      0.154598, & !  0
     &      0.133278, & !  1
     &      0.119324, & !  2
     &      0.108431, & !  3
     &      0.103499, & !  4
     &      0.100206  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 81), N1=1,  6)  &  !  SNR=   8.000
     &     / &
     &      3.500000, & !  1
     &      5.328125, & !  2
     &      7.156250, & !  3
     &      8.984375, & !  4
     &     10.812500, & !  5
     &     12.640625  & !  6
     &     /
      DATA (SPL_COE(N2, 81), N2=-2,  5) &  !  SNR=   8.000
     &     / &
     &      0.192021, & ! -2
     &      0.173453, & ! -1
     &      0.152274, & !  0
     &      0.131519, & !  1
     &      0.117910, & !  2
     &      0.107269, & !  3
     &      0.102424, & !  4
     &      0.099352  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 82), N1=1,  6)  &  !  SNR=   8.100
     &     / &
     &      3.600000, & !  1
     &      5.428125, & !  2
     &      7.256250, & !  3
     &      9.084375, & !  4
     &     10.912500, & !  5
     &     12.740625  & !  6
     &     /
      DATA (SPL_COE(N2, 82), N2=-2,  5) &  !  SNR=   8.100
     &     / &
     &      0.189139, & ! -2
     &      0.169829, & ! -1
     &      0.150157, & !  0
     &      0.129752, & !  1
     &      0.116528, & !  2
     &      0.106210, & !  3
     &      0.101048, & !  4
     &      0.098803  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 83), N1=1,  6)  &  !  SNR=   8.200
     &     / &
     &      3.700000, & !  1
     &      5.528125, & !  2
     &      7.356250, & !  3
     &      9.184375, & !  4
     &     11.012500, & !  5
     &     12.840625  & !  6
     &     /
      DATA (SPL_COE(N2, 83), N2=-2,  5) &  !  SNR=   8.200
     &     / &
     &      0.184524, & ! -2
     &      0.167327, & ! -1
     &      0.147750, & !  0
     &      0.128152, & !  1
     &      0.115176, & !  2
     &      0.105028, & !  3
     &      0.100203, & !  4
     &      0.097995  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 84), N1=1,  6)  &  !  SNR=   8.300
     &     / &
     &      3.800000, & !  1
     &      5.628125, & !  2
     &      7.456250, & !  3
     &      9.284375, & !  4
     &     11.112500, & !  5
     &     12.940625  & !  6
     &     /
      DATA (SPL_COE(N2, 84), N2=-2,  5) &  !  SNR=   8.300
     &     / &
     &      0.180479, & ! -2
     &      0.164480, & ! -1
     &      0.145518, & !  0
     &      0.126581, & !  1
     &      0.113822, & !  2
     &      0.103999, & !  3
     &      0.099124, & !  4
     &      0.096548  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 85), N1=1,  6)  &  !  SNR=   8.400
     &     / &
     &      3.900000, & !  1
     &      5.728125, & !  2
     &      7.556250, & !  3
     &      9.384375, & !  4
     &     11.212500, & !  5
     &     13.040625  & !  6
     &     /
      DATA (SPL_COE(N2, 85), N2=-2,  5) &  !  SNR=   8.400
     &     / &
     &      0.177490, & ! -2
     &      0.161662, & ! -1
     &      0.143427, & !  0
     &      0.125000, & !  1
     &      0.112522, & !  2
     &      0.102972, & !  3
     &      0.097965, & !  4
     &      0.096113  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 86), N1=1,  6)  &  !  SNR=   8.500
     &     / &
     &      4.000000, & !  1
     &      5.828125, & !  2
     &      7.656250, & !  3
     &      9.484375, & !  4
     &     11.312500, & !  5
     &     13.140625  & !  6
     &     /
      DATA (SPL_COE(N2, 86), N2=-2,  5) &  !  SNR=   8.500
     &     / &
     &      0.174736, & ! -2
     &      0.158430, & ! -1
     &      0.141695, & !  0
     &      0.123275, & !  1
     &      0.111439, & !  2
     &      0.101660, & !  3
     &      0.097341, & !  4
     &      0.094318  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 87), N1=1,  6)  &  !  SNR=   8.600
     &     / &
     &      4.100000, & !  1
     &      5.928125, & !  2
     &      7.756250, & !  3
     &      9.584375, & !  4
     &     11.412500, & !  5
     &     13.240625  & !  6
     &     /
      DATA (SPL_COE(N2, 87), N2=-2,  5) &  !  SNR=   8.600
     &     / &
     &      0.170492, & ! -2
     &      0.156492, & ! -1
     &      0.139399, & !  0
     &      0.121931, & !  1
     &      0.110115, & !  2
     &      0.100693, & !  3
     &      0.096447, & !  4
     &      0.093365  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 88), N1=1,  6)  &  !  SNR=   8.700
     &     / &
     &      4.200000, & !  1
     &      6.028125, & !  2
     &      7.856250, & !  3
     &      9.684375, & !  4
     &     11.512500, & !  5
     &     13.340625  & !  6
     &     /
      DATA (SPL_COE(N2, 88), N2=-2,  5) &  !  SNR=   8.700
     &     / &
     &      0.166380, & ! -2
     &      0.154207, & ! -1
     &      0.137383, & !  0
     &      0.120497, & !  1
     &      0.108900, & !  2
     &      0.099689, & !  3
     &      0.095433, & !  4
     &      0.092858  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 89), N1=1,  6)  &  !  SNR=   8.800
     &     / &
     &      4.300000, & !  1
     &      6.128125, & !  2
     &      7.956250, & !  3
     &      9.784375, & !  4
     &     11.612500, & !  5
     &     13.440625  & !  6
     &     /
      DATA (SPL_COE(N2, 89), N2=-2,  5) &  !  SNR=   8.800
     &     / &
     &      0.164543, & ! -2
     &      0.150869, & ! -1
     &      0.136041, & !  0
     &      0.118818, & !  1
     &      0.107864, & !  2
     &      0.098574, & !  3
     &      0.094742, & !  4
     &      0.091106  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 90), N1=1,  6)  &  !  SNR=   8.900
     &     / &
     &      4.400000, & !  1
     &      6.228125, & !  2
     &      8.056250, & !  3
     &      9.884375, & !  4
     &     11.712500, & !  5
     &     13.540625  & !  6
     &     /
      DATA (SPL_COE(N2, 90), N2=-2,  5) &  !  SNR=   8.900
     &     / &
     &      0.161982, & ! -2
     &      0.148856, & ! -1
     &      0.133942, & !  0
     &      0.117564, & !  1
     &      0.106605, & !  2
     &      0.097769, & !  3
     &      0.093422, & !  4
     &      0.091297  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 91), N1=1,  6)  &  !  SNR=   9.000
     &     / &
     &      4.500000, & !  1
     &      6.328125, & !  2
     &      8.156250, & !  3
     &      9.984375, & !  4
     &     11.812500, & !  5
     &     13.640625  & !  6
     &     /
      DATA (SPL_COE(N2, 91), N2=-2,  5) &  !  SNR=   9.000
     &     / &
     &      0.159338, & ! -2
     &      0.146831, & ! -1
     &      0.132008, & !  0
     &      0.116267, & !  1
     &      0.105458, & !  2
     &      0.096797, & !  3
     &      0.092680, & !  4
     &      0.090137  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 92), N1=1,  6)  &  !  SNR=   9.100
     &     / &
     &      4.600000, & !  1
     &      6.428125, & !  2
     &      8.256250, & !  3
     &     10.084375, & !  4
     &     11.912500, & !  5
     &     13.740625  & !  6
     &     /
      DATA (SPL_COE(N2, 92), N2=-2,  5) &  !  SNR=   9.100
     &     / &
     &      0.156598, & ! -2
     &      0.144442, & ! -1
     &      0.130447, & !  0
     &      0.114865, & !  1
     &      0.104366, & !  2
     &      0.095904, & !  3
     &      0.091700, & !  4
     &      0.089048  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 93), N1=1,  6)  &  !  SNR=   9.200
     &     / &
     &      4.700000, & !  1
     &      6.528125, & !  2
     &      8.356250, & !  3
     &     10.184375, & !  4
     &     12.012500, & !  5
     &     13.840625  & !  6
     &     /
      DATA (SPL_COE(N2, 93), N2=-2,  5) &  !  SNR=   9.200
     &     / &
     &      0.153406, & ! -2
     &      0.142965, & ! -1
     &      0.128394, & !  0
     &      0.113692, & !  1
     &      0.103286, & !  2
     &      0.094836, & !  3
     &      0.091200, & !  4
     &      0.087698  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 94), N1=1,  6)  &  !  SNR=   9.300
     &     / &
     &      4.800000, & !  1
     &      6.628125, & !  2
     &      8.456250, & !  3
     &     10.284375, & !  4
     &     12.112500, & !  5
     &     13.940625  & !  6
     &     /
      DATA (SPL_COE(N2, 94), N2=-2,  5) &  !  SNR=   9.300
     &     / &
     &      0.149936, & ! -2
     &      0.141028, & ! -1
     &      0.126707, & !  0
     &      0.112472, & !  1
     &      0.102146, & !  2
     &      0.094131, & !  3
     &      0.090003, & !  4
     &      0.088242  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 95), N1=1,  6)  &  !  SNR=   9.400
     &     / &
     &      4.900000, & !  1
     &      6.728125, & !  2
     &      8.556250, & !  3
     &     10.384375, & !  4
     &     12.212500, & !  5
     &     14.040625  & !  6
     &     /
      DATA (SPL_COE(N2, 95), N2=-2,  5) &  !  SNR=   9.400
     &     / &
     &      0.149458, & ! -2
     &      0.138383, & ! -1
     &      0.125382, & !  0
     &      0.111092, & !  1
     &      0.101188, & !  2
     &      0.093163, & !  3
     &      0.089210, & !  4
     &      0.087730  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 96), N1=1,  6)  &  !  SNR=   9.500
     &     / &
     &      5.000000, & !  1
     &      6.828125, & !  2
     &      8.656250, & !  3
     &     10.484375, & !  4
     &     12.312500, & !  5
     &     14.140625  & !  6
     &     /
      DATA (SPL_COE(N2, 96), N2=-2,  5) &  !  SNR=   9.500
     &     / &
     &      0.146700, & ! -2
     &      0.136379, & ! -1
     &      0.123848, & !  0
     &      0.109852, & !  1
     &      0.100210, & !  2
     &      0.092221, & !  3
     &      0.088645, & !  4
     &      0.085564  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 97), N1=1,  6)  &  !  SNR=   9.600
     &     / &
     &      5.100000, & !  1
     &      6.928125, & !  2
     &      8.756250, & !  3
     &     10.584375, & !  4
     &     12.412500, & !  5
     &     14.240625  & !  6
     &     /
      DATA (SPL_COE(N2, 97), N2=-2,  5) &  !  SNR=   9.600
     &     / &
     &      0.143494, & ! -2
     &      0.135233, & ! -1
     &      0.121926, & !  0
     &      0.108854, & !  1
     &      0.099090, & !  2
     &      0.091503, & !  3
     &      0.087696, & !  4
     &      0.085263  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 98), N1=1,  6)  &  !  SNR=   9.700
     &     / &
     &      5.200000, & !  1
     &      7.028125, & !  2
     &      8.856250, & !  3
     &     10.684375, & !  4
     &     12.512500, & !  5
     &     14.340625  & !  6
     &     /
      DATA (SPL_COE(N2, 98), N2=-2,  5) &  !  SNR=   9.700
     &     / &
     &      0.143253, & ! -2
     &      0.132098, & ! -1
     &      0.121103, & !  0
     &      0.107386, & !  1
     &      0.098269, & !  2
     &      0.090585, & !  3
     &      0.086823, & !  4
     &      0.085296  & !  5
     &     /
!
      DATA (SPL_ARG(N1, 99), N1=1,  6)  &  !  SNR=   9.800
     &     / &
     &      5.300000, & !  1
     &      7.128125, & !  2
     &      8.956250, & !  3
     &     10.784375, & !  4
     &     12.612500, & !  5
     &     14.440625  & !  6
     &     /
      DATA (SPL_COE(N2, 99), N2=-2,  5) &  !  SNR=   9.800
     &     / &
     &      0.139377, & ! -2
     &      0.131177, & ! -1
     &      0.119231, & !  0
     &      0.106437, & !  1
     &      0.097198, & !  2
     &      0.089871, & !  3
     &      0.085949, & !  4
     &      0.084561  & !  5
     &     /
!
      DATA (SPL_ARG(N1,100), N1=1,  6)  &  !  SNR=   9.900
     &     / &
     &      5.400000, & !  1
     &      7.228125, & !  2
     &      9.056250, & !  3
     &     10.884375, & !  4
     &     12.712500, & !  5
     &     14.540625  & !  6
     &     /
      DATA (SPL_COE(N2,100), N2=-2,  5) &  !  SNR=   9.900
     &     / &
     &      0.138237, & ! -2
     &      0.128805, & ! -1
     &      0.118179, & !  0
     &      0.105130, & !  1
     &      0.096419, & !  2
     &      0.088835, & !  3
     &      0.085645, & !  4
     &      0.082709  & !  5
     &     /
!
      DATA (SPL_ARG(N1,101), N1=1,  6)  &  !  SNR=  10.000
     &     / &
     &      5.500000, & !  1
     &      7.328125, & !  2
     &      9.156250, & !  3
     &     10.984375, & !  4
     &     12.812500, & !  5
     &     14.640625  & !  6
     &     /
      DATA (SPL_COE(N2,101), N2=-2,  5) &  !  SNR=  10.000
     &     / &
     &      0.135463, & ! -2
     &      0.127604, & ! -1
     &      0.116459, & !  0
     &      0.104209, & !  1
     &      0.095355, & !  2
     &      0.088232, & !  3
     &      0.084673, & !  4
     &      0.082493  & !  5
     &     /
!
      DATA (SPL_ARG(N1,102), N1=1,  6)  &  !  SNR=  10.100
     &     / &
     &      5.600000, & !  1
     &      7.428125, & !  2
     &      9.256250, & !  3
     &     11.084375, & !  4
     &     12.912500, & !  5
     &     14.740625  & !  6
     &     /
      DATA (SPL_COE(N2,102), N2=-2,  5) &  !  SNR=  10.100
     &     / &
     &      0.134681, & ! -2
     &      0.125820, & ! -1
     &      0.115179, & !  0
     &      0.103103, & !  1
     &      0.094500, & !  2
     &      0.087372, & !  3
     &      0.084035, & !  4
     &      0.081348  & !  5
     &     /
!
      DATA (SPL_ARG(N1,103), N1=1,  6)  &  !  SNR=  10.200
     &     / &
     &      5.700000, & !  1
     &      7.528125, & !  2
     &      9.356250, & !  3
     &     11.184375, & !  4
     &     13.012500, & !  5
     &     14.840625  & !  6
     &     /
      DATA (SPL_COE(N2,103), N2=-2,  5) &  !  SNR=  10.200
     &     / &
     &      0.132918, & ! -2
     &      0.123748, & ! -1
     &      0.114142, & !  0
     &      0.101916, & !  1
     &      0.093675, & !  2
     &      0.086605, & !  3
     &      0.083254, & !  4
     &      0.081179  & !  5
     &     /
!
      DATA (SPL_ARG(N1,104), N1=1,  6)  &  !  SNR=  10.300
     &     / &
     &      5.800000, & !  1
     &      7.628125, & !  2
     &      9.456250, & !  3
     &     11.284375, & !  4
     &     13.112500, & !  5
     &     14.940625  & !  6
     &     /
      DATA (SPL_COE(N2,104), N2=-2,  5) &  !  SNR=  10.300
     &     / &
     &      0.130422, & ! -2
     &      0.122698, & ! -1
     &      0.112590, & !  0
     &      0.101025, & !  1
     &      0.092709, & !  2
     &      0.085980, & !  3
     &      0.082402, & !  4
     &      0.080456  & !  5
     &     /
!
      DATA (SPL_ARG(N1,105), N1=1,  6)  &  !  SNR=  10.400
     &     / &
     &      5.900000, & !  1
     &      7.728125, & !  2
     &      9.556250, & !  3
     &     11.384375, & !  4
     &     13.212500, & !  5
     &     15.040625  & !  6
     &     /
      DATA (SPL_COE(N2,105), N2=-2,  5) &  !  SNR=  10.400
     &     / &
     &      0.128750, & ! -2
     &      0.121416, & ! -1
     &      0.111168, & !  0
     &      0.100093, & !  1
     &      0.091812, & !  2
     &      0.085268, & !  3
     &      0.081710, & !  4
     &      0.080030  & !  5
     &     /
!
      DATA (SPL_ARG(N1,106), N1=1,  6)  &  !  SNR=  10.500
     &     / &
     &      6.000000, & !  1
     &      7.828125, & !  2
     &      9.656250, & !  3
     &     11.484375, & !  4
     &     13.312500, & !  5
     &     15.140625  & !  6
     &     /
      DATA (SPL_COE(N2,106), N2=-2,  5) &  !  SNR=  10.500
     &     / &
     &      0.127262, & ! -2
     &      0.119695, & ! -1
     &      0.110074, & !  0
     &      0.099007, & !  1
     &      0.091077, & !  2
     &      0.084347, & !  3
     &      0.081400, & !  4
     &      0.078828  & !  5
     &     /
!
      DATA (SPL_ARG(N1,107), N1=1,  6)  &  !  SNR=  10.600
     &     / &
     &      6.100000, & !  1
     &      7.928125, & !  2
     &      9.756250, & !  3
     &     11.584375, & !  4
     &     13.412500, & !  5
     &     15.240625  & !  6
     &     /
      DATA (SPL_COE(N2,107), N2=-2,  5) &  !  SNR=  10.600
     &     / &
     &      0.125995, & ! -2
     &      0.118037, & ! -1
     &      0.108988, & !  0
     &      0.097983, & !  1
     &      0.090295, & !  2
     &      0.083610, & !  3
     &      0.080697, & !  4
     &      0.078421  & !  5
     &     /
!
      DATA (SPL_ARG(N1,108), N1=1,  6)  &  !  SNR=  10.700
     &     / &
     &      6.200000, & !  1
     &      8.028125, & !  2
     &      9.856250, & !  3
     &     11.684375, & !  4
     &     13.512500, & !  5
     &     15.340625  & !  6
     &     /
      DATA (SPL_COE(N2,108), N2=-2,  5) &  !  SNR=  10.700
     &     / &
     &      0.123593, & ! -2
     &      0.117005, & ! -1
     &      0.107686, & !  0
     &      0.097083, & !  1
     &      0.089466, & !  2
     &      0.082904, & !  3
     &      0.080125, & !  4
     &      0.077151  & !  5
     &     /
!
      DATA (SPL_ARG(N1,109), N1=1,  6)  &  !  SNR=  10.800
     &     / &
     &      6.300000, & !  1
     &      8.128125, & !  2
     &      9.956250, & !  3
     &     11.784375, & !  4
     &     13.612500, & !  5
     &     15.440625  & !  6
     &     /
      DATA (SPL_COE(N2,109), N2=-2,  5) &  !  SNR=  10.800
     &     / &
     &      0.122061, & ! -2
     &      0.115341, & ! -1
     &      0.106608, & !  0
     &      0.096140, & !  1
     &      0.088634, & !  2
     &      0.082337, & !  3
     &      0.079234, & !  4
     &      0.077282  & !  5
     &     /
!
      DATA (SPL_ARG(N1,110), N1=1,  6)  &  !  SNR=  10.900
     &     / &
     &      6.400000, & !  1
     &      8.228125, & !  2
     &     10.056250, & !  3
     &     11.884375, & !  4
     &     13.712500, & !  5
     &     15.540625  & !  6
     &     /
      DATA (SPL_COE(N2,110), N2=-2,  5) &  !  SNR=  10.900
     &     / &
     &      0.120817, & ! -2
     &      0.114060, & ! -1
     &      0.105419, & !  0
     &      0.095240, & !  1
     &      0.087873, & !  2
     &      0.081603, & !  3
     &      0.078686, & !  4
     &      0.076663  & !  5
     &     /
!
      DATA (SPL_ARG(N1,111), N1=1,  6)  &  !  SNR=  11.000
     &     / &
     &      6.500000, & !  1
     &      8.328125, & !  2
     &     10.156250, & !  3
     &     11.984375, & !  4
     &     13.812500, & !  5
     &     15.640625  & !  6
     &     /
      DATA (SPL_COE(N2,111), N2=-2,  5) &  !  SNR=  11.000
     &     / &
     &      0.120156, & ! -2
     &      0.112327, & ! -1
     &      0.104520, & !  0
     &      0.094283, & !  1
     &      0.087079, & !  2
     &      0.081072, & !  3
     &      0.077757, & !  4
     &      0.076865  & !  5
     &     /
!
      DATA (SPL_ARG(N1,112), N1=1,  6)  &  !  SNR=  11.100
     &     / &
     &      6.600000, & !  1
     &      8.428125, & !  2
     &     10.256250, & !  3
     &     12.084375, & !  4
     &     13.912500, & !  5
     &     15.740625  & !  6
     &     /
      DATA (SPL_COE(N2,112), N2=-2,  5) &  !  SNR=  11.100
     &     / &
     &      0.117051, & ! -2
     &      0.111751, & ! -1
     &      0.103124, & !  0
     &      0.093510, & !  1
     &      0.086326, & !  2
     &      0.080312, & !  3
     &      0.077403, & !  4
     &      0.075404  & !  5
     &     /
!
      DATA (SPL_ARG(N1,113), N1=1,  6)  &  !  SNR=  11.200
     &     / &
     &      6.700000, & !  1
     &      8.528125, & !  2
     &     10.356250, & !  3
     &     12.184375, & !  4
     &     14.012500, & !  5
     &     15.840625  & !  6
     &     /
      DATA (SPL_COE(N2,113), N2=-2,  5) &  !  SNR=  11.200
     &     / &
     &      0.117259, & ! -2
     &      0.109467, & ! -1
     &      0.102564, & !  0
     &      0.092432, & !  1
     &      0.085690, & !  2
     &      0.079625, & !  3
     &      0.076813, & !  4
     &      0.075037  & !  5
     &     /
!
      DATA (SPL_ARG(N1,114), N1=1,  6)  &  !  SNR=  11.300
     &     / &
     &      6.800000, & !  1
     &      8.628125, & !  2
     &     10.456250, & !  3
     &     12.284375, & !  4
     &     14.112500, & !  5
     &     15.940625  & !  6
     &     /
      DATA (SPL_COE(N2,114), N2=-2,  5) &  !  SNR=  11.300
     &     / &
     &      0.114117, & ! -2
     &      0.109387, & ! -1
     &      0.100960, & !  0
     &      0.091836, & !  1
     &      0.084826, & !  2
     &      0.079079, & !  3
     &      0.076159, & !  4
     &      0.074612  & !  5
     &     /
!
      DATA (SPL_ARG(N1,115), N1=1,  6)  &  !  SNR=  11.400
     &     / &
     &      6.900000, & !  1
     &      8.728125, & !  2
     &     10.556250, & !  3
     &     12.384375, & !  4
     &     14.212500, & !  5
     &     16.040625  & !  6
     &     /
      DATA (SPL_COE(N2,115), N2=-2,  5) &  !  SNR=  11.400
     &     / &
     &      0.113765, & ! -2
     &      0.107596, & ! -1
     &      0.100230, & !  0
     &      0.090877, & !  1
     &      0.084166, & !  2
     &      0.078456, & !  3
     &      0.075469, & !  4
     &      0.074224  & !  5
     &     /
!
      DATA (SPL_ARG(N1,116), N1=1,  6)  &  !  SNR=  11.500
     &     / &
     &      7.000000, & !  1
     &      8.828125, & !  2
     &     10.656250, & !  3
     &     12.484375, & !  4
     &     14.312500, & !  5
     &     16.140625  & !  6
     &     /
      DATA (SPL_COE(N2,116), N2=-2,  5) &  !  SNR=  11.500
     &     / &
     &      0.112484, & ! -2
     &      0.106239, & ! -1
     &      0.099328, & !  0
     &      0.090010, & !  1
     &      0.083495, & !  2
     &      0.077816, & !  3
     &      0.074988, & !  4
     &      0.073417  & !  5
     &     /
!
      DATA (SPL_ARG(N1,117), N1=1,  6)  &  !  SNR=  11.600
     &     / &
     &      7.100000, & !  1
     &      8.928125, & !  2
     &     10.756250, & !  3
     &     12.584375, & !  4
     &     14.412500, & !  5
     &     16.240625  & !  6
     &     /
      DATA (SPL_COE(N2,117), N2=-2,  5) &  !  SNR=  11.600
     &     / &
     &      0.110663, & ! -2
     &      0.105700, & ! -1
     &      0.097988, & !  0
     &      0.089378, & !  1
     &      0.082710, & !  2
     &      0.077263, & !  3
     &      0.074378, & !  4
     &      0.072861  & !  5
     &     /
!
      DATA (SPL_ARG(N1,118), N1=1,  6)  &  !  SNR=  11.700
     &     / &
     &      7.200000, & !  1
     &      9.028125, & !  2
     &     10.856250, & !  3
     &     12.684375, & !  4
     &     14.512500, & !  5
     &     16.340625  & !  6
     &     /
      DATA (SPL_COE(N2,118), N2=-2,  5) &  !  SNR=  11.700
     &     / &
     &      0.109123, & ! -2
     &      0.104467, & ! -1
     &      0.097125, & !  0
     &      0.088542, & !  1
     &      0.082054, & !  2
     &      0.076658, & !  3
     &      0.073841, & !  4
     &      0.072774  & !  5
     &     /
!
      DATA (SPL_ARG(N1,119), N1=1,  6)  &  !  SNR=  11.800
     &     / &
     &      7.300000, & !  1
     &      9.128125, & !  2
     &     10.956250, & !  3
     &     12.784375, & !  4
     &     14.612500, & !  5
     &     16.440625  & !  6
     &     /
      DATA (SPL_COE(N2,119), N2=-2,  5) &  !  SNR=  11.800
     &     / &
     &      0.108556, & ! -2
     &      0.103175, & ! -1
     &      0.096291, & !  0
     &      0.087729, & !  1
     &      0.081387, & !  2
     &      0.076115, & !  3
     &      0.073186, & !  4
     &      0.072089  & !  5
     &     /
!
      DATA (SPL_ARG(N1,120), N1=1,  6)  &  !  SNR=  11.900
     &     / &
     &      7.400000, & !  1
     &      9.228125, & !  2
     &     11.056250, & !  3
     &     12.884375, & !  4
     &     14.712500, & !  5
     &     16.540625  & !  6
     &     /
      DATA (SPL_COE(N2,120), N2=-2,  5) &  !  SNR=  11.900
     &     / &
     &      0.107334, & ! -2
     &      0.102025, & ! -1
     &      0.095404, & !  0
     &      0.086947, & !  1
     &      0.080755, & !  2
     &      0.075492, & !  3
     &      0.072654, & !  4
     &      0.071605  & !  5
     &     /
!
      DATA (SPL_ARG(N1,121), N1=1,  6)  &  !  SNR=  12.000
     &     / &
     &      7.500000, & !  1
     &      9.328125, & !  2
     &     11.156250, & !  3
     &     12.984375, & !  4
     &     14.812500, & !  5
     &     16.640625  & !  6
     &     /
      DATA (SPL_COE(N2,121), N2=-2,  5) &  !  SNR=  12.000
     &     / &
     &      0.106434, & ! -2
     &      0.100693, & ! -1
     &      0.094654, & !  0
     &      0.086125, & !  1
     &      0.080143, & !  2
     &      0.074907, & !  3
     &      0.072189, & !  4
     &      0.070878  & !  5
     &     /
!
      DATA (SPL_ARG(N1,122), N1=1,  6)  &  !  SNR=  12.100
     &     / &
     &      7.600000, & !  1
     &      9.428125, & !  2
     &     11.256250, & !  3
     &     13.084375, & !  4
     &     14.912500, & !  5
     &     16.740625  & !  6
     &     /
      DATA (SPL_COE(N2,122), N2=-2,  5) &  !  SNR=  12.100
     &     / &
     &      0.105581, & ! -2
     &      0.099778, & ! -1
     &      0.093672, & !  0
     &      0.085434, & !  1
     &      0.079486, & !  2
     &      0.074354, & !  3
     &      0.071680, & !  4
     &      0.070372  & !  5
     &     /
!
      DATA (SPL_ARG(N1,123), N1=1,  6)  &  !  SNR=  12.200
     &     / &
     &      7.700000, & !  1
     &      9.528125, & !  2
     &     11.356250, & !  3
     &     13.184375, & !  4
     &     15.012500, & !  5
     &     16.840625  & !  6
     &     /
      DATA (SPL_COE(N2,123), N2=-2,  5) &  !  SNR=  12.200
     &     / &
     &      0.103450, & ! -2
     &      0.099198, & ! -1
     &      0.092620, & !  0
     &      0.084791, & !  1
     &      0.078801, & !  2
     &      0.073862, & !  3
     &      0.071083, & !  4
     &      0.069870  & !  5
     &     /
!
      DATA (SPL_ARG(N1,124), N1=1,  6)  &  !  SNR=  12.300
     &     / &
     &      7.800000, & !  1
     &      9.628125, & !  2
     &     11.456250, & !  3
     &     13.284375, & !  4
     &     15.112500, & !  5
     &     16.940625  & !  6
     &     /
      DATA (SPL_COE(N2,124), N2=-2,  5) &  !  SNR=  12.300
     &     / &
     &      0.103237, & ! -2
     &      0.097786, & ! -1
     &      0.091971, & !  0
     &      0.083948, & !  1
     &      0.078344, & !  2
     &      0.073026, & !  3
     &      0.071053, & !  4
     &      0.068770  & !  5
     &     /
!
      DATA (SPL_ARG(N1,125), N1=1,  6)  &  !  SNR=  12.400
     &     / &
     &      7.900000, & !  1
     &      9.728125, & !  2
     &     11.556250, & !  3
     &     13.384375, & !  4
     &     15.212500, & !  5
     &     17.040625  & !  6
     &     /
      DATA (SPL_COE(N2,125), N2=-2,  5) &  !  SNR=  12.400
     &     / &
     &      0.101702, & ! -2
     &      0.097036, & ! -1
     &      0.091027, & !  0
     &      0.083337, & !  1
     &      0.077608, & !  2
     &      0.072748, & !  3
     &      0.070083, & !  4
     &      0.068842  & !  5
     &     /
!
      DATA (SPL_ARG(N1,126), N1=1,  6)  &  !  SNR=  12.500
     &     / &
     &      8.000000, & !  1
     &      9.828125, & !  2
     &     11.656250, & !  3
     &     13.484375, & !  4
     &     15.312500, & !  5
     &     17.140625  & !  6
     &     /
      DATA (SPL_COE(N2,126), N2=-2,  5) &  !  SNR=  12.500
     &     / &
     &      0.100495, & ! -2
     &      0.096061, & ! -1
     &      0.090243, & !  0
     &      0.082617, & !  1
     &      0.077042, & !  2
     &      0.072171, & !  3
     &      0.069690, & !  4
     &      0.068226  & !  5
     &     /
!
      DATA (SPL_ARG(N1,127), N1=1,  6)  &  !  SNR=  12.600
     &     / &
     &      8.100000, & !  1
     &      9.928125, & !  2
     &     11.756250, & !  3
     &     13.584375, & !  4
     &     15.412500, & !  5
     &     17.240625  & !  6
     &     /
      DATA (SPL_COE(N2,127), N2=-2,  5) &  !  SNR=  12.600
     &     / &
     &      0.099592, & ! -2
     &      0.095204, & ! -1
     &      0.089351, & !  0
     &      0.081977, & !  1
     &      0.076446, & !  2
     &      0.071630, & !  3
     &      0.069290, & !  4
     &      0.068079  & !  5
     &     /
!
      DATA (SPL_ARG(N1,128), N1=1,  6)  &  !  SNR=  12.700
     &     / &
     &      8.200000, & !  1
     &     10.028125, & !  2
     &     11.856250, & !  3
     &     13.684375, & !  4
     &     15.512500, & !  5
     &     17.340625  & !  6
     &     /
      DATA (SPL_COE(N2,128), N2=-2,  5) &  !  SNR=  12.700
     &     / &
     &      0.098832, & ! -2
     &      0.094423, & ! -1
     &      0.088483, & !  0
     &      0.081353, & !  1
     &      0.075825, & !  2
     &      0.071182, & !  3
     &      0.068724, & !  4
     &      0.067685  & !  5
     &     /
!
      REAL*8     AMP_USE, SIG_PHS(2), EPS
      PARAMETER  ( EPS = 1.D-5 )
      INTEGER*4  IND_SNR(2), J1
      REAL*8,    EXTERNAL :: EBSPL_VAL_R8 
!
      IF ( SNR .LE. SNR_MIN ) THEN
           PIMA_AMP_TO_SIGPHS  = PI__NUM/DSQRT(3.0D0)
        ELSE IF ( SNR .GE. SNR_MAX ) THEN
           IF ( AMP > AMP_MIN ) THEN
                PIMA_AMP_TO_SIGPHS = 1.0D0/AMP
             ELSE 
                PIMA_AMP_TO_SIGPHS = PI__NUM/DSQRT(3.0D0)
           END IF
        ELSE 
           IND_SNR(1) = (SNR - SNR_MIN)/SNR_STP + 1
           IND_SNR(2) = IND_SNR(1) + 1
           IF ( IND_SNR(2) >  MS ) IND_SNR(2) = MS
           DO 410 J1=1,2
              AMP_USE = AMP
              IF ( AMP_USE < SPL_ARG(1,IND_SNR(J1)) + EPS ) THEN
                   PIMA_AMP_TO_SIGPHS = PI__NUM/DSQRT(3.0D0)
                   RETURN 
              END IF
              IF ( AMP_USE > SPL_ARG(MN,IND_SNR(J1)) - EPS ) THEN
                   AMP_USE = SPL_ARG(MN,IND_SNR(J1)) - EPS
              END IF
              SIG_PHS(J1) = EBSPL_VAL_R8 ( MN, DEG, AMP_USE, &
     &                                     SPL_ARG(1,IND_SNR(J1)), &
     &                                     SPL_COE(1-DEG,IND_SNR(J1)) ) 
 410       CONTINUE 
           PIMA_AMP_TO_SIGPHS = SIG_PHS(1) + (SIG_PHS(2) - SIG_PHS(1))* &
     &                          (SNR - SNR_MIN - (IND_SNR(1)-1)*SNR_STP)
!%  write ( 6, * ) ' snr= ', snr, ' ind_snr= ', ind_snr  ! %%%
!%  write ( 6, * ) ' s1= ', (SIG_PHS(2) - SIG_PHS(1)), ' s2 =', (SNR - SNR_MIN - (IND_SNR(1)-1)*SNR_STP) ! %%%
!%  write ( 6, * ) ' sig_phs = ', sig_phs, ' pi= ', pima_amp_to_sigphs ! %%%%%5
      END IF
!
      RETURN
      END  FUNCTION  PIMA_AMP_TO_SIGPHS  !#!  
!
! ------------------------------------------------------------------------
!
      FUNCTION   PIMA_SIG_PHS ( SNR )
! ************************************************************************
! *                                                                      *
! *   Routine  PIMA_SIG_PHS  returns the variance of the fringe phase as *
! *   a function of elementary SNR. Error of interpolation: 0.001        *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ### 25-MAR-2010  PIMA_SIG_PHS  v1.0 (d) L. Petrov  05-APR-2010 ###  *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      REAL*8     PIMA_SIG_PHS
      REAL*8     SNR
      INTEGER*4    DEG, MN
      PARAMETER  ( DEG = 3  )
      PARAMETER  ( MN  = 14 )
      REAL*8     SIG_PHS_0, SNR_ARG(1:MN), SIG_SPL(1-DEG:MN-1)
      PARAMETER  ( SIG_PHS_0 = 0.0D0 )
      DATA       SNR_ARG / &
     &           0.004990, & !  1
     &           0.792308, & !  2
     &           1.579615, & !  3
     &           2.366923, & !  4
     &           3.154231, & !  5
     &           3.941538, & !  6
     &           4.728846, & !  7
     &           5.516154, & !  8
     &           6.303462, & !  9
     &           7.090769, & ! 10
     &           7.878077, & ! 11
     &           8.665385, & ! 12
     &           9.452692, & ! 13
     &          10.240010  & ! 14
     &                   /
!
      DATA       SIG_SPL / &
     &           0.008777, & ! -2
     &           0.485394, & ! -1
     &           1.129385, & !  0
     &           1.291407, & !  1
     &           1.153665, & !  2
     &           1.067533, & !  3
     &           1.037399, & !  4
     &           1.024529, & !  5
     &           1.017529, & !  6
     &           1.013195, & !  7
     &           1.010309, & !  8
     &           1.008298, & !  9
     &           1.006800, & ! 10
     &           1.005709, & ! 11
     &           1.005124, & ! 12
     &           1.004861  & ! 13
     &                   /
      REAL*8,    EXTERNAL :: EBSPL_VAL_R8 
!
      IF ( SNR .LE. SNR_ARG(1) )  THEN
           PIMA_SIG_PHS = SIG_PHS_0 
         ELSE IF ( SNR .GE. SNR_ARG(MN) ) THEN
           PIMA_SIG_PHS = 1.D0/SNR
         ELSE
           PIMA_SIG_PHS = EBSPL_VAL_R8 ( MN, DEG, SNR, SNR_ARG, SIG_SPL )/SNR
      END IF
!
      RETURN
      END  FUNCTION   PIMA_SIG_PHS   !#!  
!
! ------------------------------------------------------------------------
!
      FUNCTION   PIMA_SIG_AMP ( SNR )
! ************************************************************************
! *                                                                      *
! *   Routine  PIMA_SIG_AMP  returns the variance of the fringe phase as *
! *   a function of elementary SNR. Interpolation error: 0.0005          *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ### 25-MAR-2010  PIMA_SIG_AMP  v1.0 (d) L. Petrov  05-APR-2010 ###  *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      REAL*8     PIMA_SIG_AMP
      REAL*8     SNR
      INTEGER*4    DEG, MN
      PARAMETER  ( DEG = 3  )
      PARAMETER  ( MN  = 14 )
      REAL*8       PI__NUM
      PARAMETER  ( PI__NUM = 3.141592653589793D0 )
      REAL*8     SIG_AMP_0, SNR_ARG(1:MN), SIG_SPL(1-DEG:MN-1)
      PARAMETER  ( SIG_AMP_0 = DSQRT ( 2.0D0 ) )
      DATA       SNR_ARG / &
     &           0.004990, & !  1
     &           0.792308, & !  2
     &           1.579615, & !  3
     &           2.366923, & !  4
     &           3.154231, & !  5
     &           3.941538, & !  6
     &           4.728846, & !  7
     &           5.516154, & !  8
     &           6.303462, & !  9
     &           7.090769, & ! 10
     &           7.878077, & ! 11
     &           8.665385, & ! 12
     &           9.452692, & ! 13
     &          10.240010  & ! 14
     &                   /
!
      DATA       SIG_SPL / &
     &           1.414821, & ! -2
     &           1.412833, & ! -1
     &           1.573644, & !  0
     &           2.100186, & !  1
     &           2.747786, & !  2
     &           3.452036, & !  3
     &           4.184791, & !  4
     &           4.934130, & !  5
     &           5.693442, & !  6
     &           6.459404, & !  7
     &           7.229879, & !  8
     &           8.003591, & !  9
     &           8.779727, & ! 10
     &           9.557658, & ! 11
     &          10.077232, & ! 12
     &          10.337202  & ! 13
     &                   /
      REAL*8,    EXTERNAL :: EBSPL_VAL_R8 
!
      IF ( SNR .LE. SNR_ARG(1) )  THEN
           PIMA_SIG_AMP = SIG_AMP_0 
         ELSE IF ( SNR .GE. SNR_ARG(MN) ) THEN
           PIMA_SIG_AMP = 1.D0/SNR
         ELSE
           PIMA_SIG_AMP = EBSPL_VAL_R8 ( MN, DEG, SNR, SNR_ARG, SIG_SPL )/SNR
      END IF
!
      RETURN
      END  FUNCTION   PIMA_SIG_AMP   !#!  
!
! ------------------------------------------------------------------------
!
      FUNCTION   PIMA_AVR_AMP ( SNR )
! ************************************************************************
! *                                                                      *
! *   Routine  PIMA_AVR_AMP  returns the variance of the fringe phase as *
! *   a function of elementary SNR. Interpolation error: 0.0005          *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ### 25-MAR-2010  PIMA_AVR_AMP  v1.0 (d) L. Petrov  05-APR-2010 ###  *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      REAL*8     PIMA_AVR_AMP
      REAL*8     SNR
      INTEGER*4    DEG, MN
      PARAMETER  ( DEG = 3  )
      PARAMETER  ( MN  = 14 )
      REAL*8       PI__NUM
      PARAMETER  ( PI__NUM = 3.141592653589793D0 )
      REAL*8     AVR_AMP_0, SNR_ARG(1:MN), SIG_SPL(1-DEG:MN-1)
      PARAMETER  ( AVR_AMP_0 = DSQRT ( PI__NUM/2.0D0 ) )
      DATA       SNR_ARG / &
     &           0.004990, & !  1
     &           0.792308, & !  2
     &           1.579615, & !  3
     &           2.366923, & !  4
     &           3.154231, & !  5
     &           3.941538, & !  6
     &           4.728846, & !  7
     &           5.516154, & !  8
     &           6.303462, & !  9
     &           7.090769, & ! 10
     &           7.878077, & ! 11
     &           8.665385, & ! 12
     &           9.452692, & ! 13
     &          10.240010  & ! 14
     &                   /
!
      DATA       SIG_SPL / &
     &           1.253520, & ! -2
     &           1.253133, & ! -1
     &           1.390675, & !  0
     &           1.908533, & !  1
     &           2.581978, & !  2
     &           3.313840, & !  3
     &           4.068888, & !  4
     &           4.834822, & !  5
     &           5.606955, & !  6
     &           6.382876, & !  7
     &           7.161353, & !  8
     &           7.941577, & !  9
     &           8.723124, & ! 10
     &           9.505618, & ! 11
     &          10.027767, & ! 12
     &          10.288970  & ! 13
     &                   /
      REAL*8,    EXTERNAL :: EBSPL_VAL_R8 
!
      IF ( SNR .LE. SNR_ARG(1) )  THEN
           PIMA_AVR_AMP = AVR_AMP_0 
         ELSE IF ( SNR .GE. SNR_ARG(MN) ) THEN
           PIMA_AVR_AMP = 1.D0/SNR
         ELSE
           PIMA_AVR_AMP = EBSPL_VAL_R8 ( MN, DEG, SNR, SNR_ARG, SIG_SPL )/SNR
      END IF
!
      RETURN
      END  FUNCTION   PIMA_AVR_AMP   !#!  
